# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.library.python.common23.models.core.geo.code_system import CodeSystem
from travel.rasp.library.python.common23.models.fields import CodeCharField


class StationCode(models.Model):
    u"""Код станции"""
    station = models.ForeignKey(Station, verbose_name=_(u'станция'), blank=False,
                                null=False, related_name='code_set')
    code = CodeCharField(verbose_name=_(u"код"), blank=False, null=False,
                         max_length=50, db_index=True)
    system = models.ForeignKey(CodeSystem, verbose_name=_(u'система кодирования'),
                               blank=False, null=False)

    @classmethod
    def id_to_code_dict(cls, system):
        return dict(
            (sc.station_id, sc.code)
            for sc in cls.objects.filter(system__code=system)
        )

    @classmethod
    def id_to_code_dict_for(cls, system, stations):
        cs = CodeSystem.objects.get(code=system)

        return dict(
            (sc.station_id, sc.code)
            for sc in cls.objects.filter(system=cs, station__in=stations)
        )

    class StationCodeByStationIdGetter(object):
        """
        Класс при инициализации получает одним запросом
        данные о кодах переданных станций в заданной системе кодирования.
        И позволяет получать код станции по ее id.
        """

        def __init__(self, system_id, station_ids, default=None):
            self.default = default

            # order_by() для того, чтобы сбросить сортировку по-умолчанию и избавится от JOIN
            self.station_code_by_station_id = dict(
                StationCode.objects.filter(system_id=system_id, station_id__in=station_ids)
                    .order_by()
                    .values_list('station_id', 'code')
            )

        def get(self, station_id):
            return self.station_code_by_station_id.get(station_id, self.default)

    def __unicode__(self):
        return self.station.title + u': ' + self.system.code + u' = ' + self.code

    class Meta(object):
        verbose_name = _(u'код станции')
        verbose_name_plural = _(u'коды станций')
        # В каждой системе кодирования станции имеет только один код
        unique_together = (('station', 'system'), ('system', 'code'))
        app_label = 'www'
        db_table = 'www_stationcode'
