# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from contextlib import contextmanager

from django.db import models


class StationCodeManager(models.Manager):
    station_cache = None
    code_cache = None

    def __init__(self):
        models.Manager.__init__(self)

    def get_list_by_code(self, code):
        from travel.rasp.library.python.common23.models.core.geo.station import Station

        if StationCodeManager.station_cache is not None:
            pks = StationCodeManager.station_cache.get(code.lower(), [])

            return self.model.objects.in_bulk_list(pks)

        return list(Station.objects.filter(hidden=False, code_set__code=code))

    def getcodes(self, station):
        from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode

        if StationCodeManager.code_cache is not None:
            return StationCodeManager.code_cache.get(station.id, {})

        return dict(
            (sc.system.code, sc.code)
            for sc in StationCode.objects.filter(station=station)
        )

    @classmethod
    def precache(cls, systems=None):
        from travel.rasp.library.python.common23.models.core.geo.station_code import StationCode
        from travel.rasp.library.python.common23.models.core.geo.code_system import CodeSystem

        station_cache = {}
        code_cache = {}

        codesystems = dict(
            (cs.id, cs.code)
            for cs in CodeSystem.objects.all()
        )

        if not systems:
            systems = list(codesystems.values())

        for sc in StationCode.objects.filter(station__hidden=False, system__code__in=systems):
            key = sc.code.lower()

            station_cache.setdefault(key, []).append(sc.station_id)

            system_code = codesystems[sc.system_id]

            code_cache.setdefault(sc.station_id, {})[system_code] = sc.code

        cls.station_cache = station_cache
        cls.code_cache = code_cache

    @contextmanager
    def using_precache(self):
        if StationCodeManager.station_cache is None:
            self.precache()
            try:
                yield
            finally:
                StationCodeManager.station_cache = StationCodeManager.code_cache = None
        else:
            yield
