# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import str
from lxml import etree

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.fields import CodeCharField, TrimmedCharField
from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager


class StationTerminal(models.Model):
    u"""Терминал станции (аэропорта)"""

    name = TrimmedCharField(_(u'Название'), max_length=50, blank=False, null=False)
    station = models.ForeignKey(Station, verbose_name=_(u'станция'), null=False, blank=False)
    sirena_id = CodeCharField(_(u'Диана код терминала'), max_length=6,
                              blank=True, null=True)
    sirena_letter = CodeCharField(_(u'Сирена буква терминала'), max_length=6,
                                  blank=True, null=True)

    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None,
                                  blank=True, null=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None,
                                 blank=True, null=True)

    is_international = models.BooleanField(verbose_name=_(u'Является международным'), default=False)
    is_domestic = models.BooleanField(verbose_name=_(u'Является внутренним'), default=False)

    objects = PrecachingManager(keys=['pk', 'station'])

    def __unicode__(self):
        return "%s-%s" % (self.station.title or '', self.name)

    @property
    def export_uid(self):
        u"""ID терминала для экспорта"""
        return "exit__%d" % self.id

    @property
    def export_type(self):
        u"""Тип терминала для экспорта"""
        return "exit"

    def as_xml(self):
        element = etree.Element('terminal',
                                name=self.name,
                                lat=str(self.latitude),
                                lon=str(self.longitude))

        return element

    class Meta(object):
        verbose_name = _(u'терминал')
        verbose_name_plural = _(u'терминалы')
        app_label = 'www'
        db_table = 'www_stationterminal'

    @classmethod
    def get_airport_terminal(cls, airport, letter):
        """
        Правильно работает только для аэропортов
        """
        if not letter:
            return

        try:
            return airport.stationterminal_set.get(name=letter)
        except StationTerminal.DoesNotExist:
            pass

        for terminal in airport.stationterminal_set.all():
            if terminal.sirena_letter:
                for sirena_letter in terminal.sirena_letter.split(u'/'):
                    if sirena_letter.upper() == letter.upper():
                        return terminal
