# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re

from django.db import models
from django.utils.translation import ugettext_lazy as _
from travel.rasp.library.python.common23.models.fields import RegExpField

from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field


class StationType(models.Model):
    """ Тип станции """

    TANKER_L_FIELDS = ['name', 'prefix', 'railway_prefix']

    L_name = L_field(_(u'название'), max_length=255, base_field_critical=True, base_lang='ru', extra={
        'ru': [
            ('genitive', L_field.char_field(_(u'название в родительном падеже'))),  # name_gen
            ('accusative', L_field.char_field(_(u'название в винительном падеже'))),  # name_acc
            ('dative', L_field.char_field(_(u'название в дательном падеже'))),  # name_dat
        ]
    })

    L_prefix = L_field(_(u'префикс'), base_lang='ru', max_length=100)

    L_railway_prefix = L_field(_(u'префикс для жд'), base_lang='ru', max_length=100,
                               help_text=_(u"для списка станций на странице жд рейса"))

    title_regex = RegExpField(verbose_name=_(u'шаблон названия станции'), null=True, blank=True,
                              default=None, max_length=100)

    default_for_t_type = models.OneToOneField('www.TransportType', null=True,
                                              verbose_name=_(u"по умочанию для типа транспорта"),
                                              default=None, blank=True, related_name='default_station_type')

    types = None
    regexes = None

    objects = PrecachingManager(keys=['pk'])

    STATION_ID = 1
    PLATFORM_ID = 2
    STOP_ID = 3
    CHECKPOINT_ID = 4
    POST_ID = 5
    CROSSING_ID = 6
    OVERTAKING_POINT_ID = 7
    TRAIN_STATION_ID = 8
    AIRPORT_ID = 9
    BUS_STATION_ID = 10
    BUS_STOP_ID = 11
    UNKNOWN_ID = 12
    PORT_ID = 13
    PORT_POINT_ID = 14
    WHARF_ID = 15
    RIVER_PORT_ID = 16
    MARINE_STATION_ID = 17

    STATION_TYPE_ID_BY_CODE = {
        'station': STATION_ID,
        'platform': PLATFORM_ID,
        'stop': STOP_ID,
        'checkpoint': CHECKPOINT_ID,
        'post': POST_ID,
        'crossing': CROSSING_ID,
        'overtaking_point': OVERTAKING_POINT_ID,
        'train_station': TRAIN_STATION_ID,
        'airport': AIRPORT_ID,
        'bus_station': BUS_STATION_ID,
        'bus_stop': BUS_STOP_ID,
        'unknown': UNKNOWN_ID,
        'port': PORT_ID,
        'port_point': PORT_POINT_ID,
        'wharf': WHARF_ID,
        'river_port': RIVER_PORT_ID,
        'marine_station': MARINE_STATION_ID,
    }

    STATION_TYPE_CODE_BY_ID = {v: k for k, v in STATION_TYPE_ID_BY_CODE.items()}

    @classmethod
    def types_map(cls):
        if not cls.types:
            cls.types = {t.name_ru: t for t in cls.objects.all()}

        return cls.types

    @classmethod
    def regex_list(cls):
        if not cls.regexes:
            cls.regexes = [
                (t, re.compile(t.title_regex, re.U | re.I))
                for t in cls.objects.filter(title_regex__isnull=False).exclude(title_regex='')
            ]

        return cls.regexes

    def __unicode__(self):
        return self.L_name()

    @property
    def L_title(self):
        return self.L_name

    @property
    def code(self):
        """
        deprecated. Лучше использовать id типа
        """
        return self.STATION_TYPE_CODE_BY_ID.get(self.id, '')

    class Meta(object):
        verbose_name = _(u'тип станции')
        verbose_name_plural = _(u'типы станций')
        app_label = 'www'
        db_table = 'www_stationtype'
