# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import pickle
import zlib

from django.conf import settings
from django.core.cache import cache
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.core.directions.external_direction_marker import ExternalDirectionMarker
from travel.rasp.library.python.common23.models.core.geo.settlement import Settlement
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field


class SuburbanZone(models.Model):
    """ Пригородная зона """

    L_title = L_field(verbose_name=_(u"название"), max_length=200,
                      add_local_field=True, local_field_critical=True)
    code = models.CharField(verbose_name=_(u"код зоны"),
                            max_length=100, blank=False, null=False,
                            unique=True)
    settlement = models.ForeignKey(Settlement, verbose_name=_(u"основной город зоны"),
                                   related_name="zone_settlement")
    title_from = models.CharField(verbose_name=_(u'название направления туда(от Москвы)'),
                                  max_length=100, blank=False)
    title_to = models.CharField(verbose_name=_(u'название направления обратно(на Москву)'),
                                max_length=100, blank=False)

    majority = models.IntegerField(verbose_name=_(u'важность'), default=10)

    objects = PrecachingManager()

    TANKER_L_FIELDS = ['title']

    def __unicode__(self):
        return self.title + u' <' + self.code + u'>'

    def markers(self):
        cache_key = settings.CACHEROOT + 'zone/%s/markers' % self.pk

        compressed = cache.get(cache_key)
        if compressed:
            pickled = zlib.decompress(compressed)
            return pickle.loads(pickled)

        markers = ExternalDirectionMarker.objects \
            .filter(external_direction__suburban_zone=self) \
            .select_related('external_direction') \
            .order_by()
        markers = list(markers)

        pickled = pickle.dumps(markers, protocol=pickle.HIGHEST_PROTOCOL)
        compressed = zlib.compress(pickled)
        cache.set(cache_key, compressed, settings.CACHES['default']['LONG_TIMEOUT'])

        return markers

    class Meta(object):
        verbose_name = _(u'пригородная зона')
        verbose_name_plural = _(u'пригородные зоны')
        app_label = 'www'
        db_table = 'www_suburbanzone'
        ordering = ('title',)
