# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
from string import Formatter

from django.core.exceptions import ValidationError
from django.core.validators import RegexValidator
from django.db import models
from django.utils.deconstruct import deconstructible
from django.utils.http import urlquote
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.db.mds.s3_storage import mds_s3_media_storage
from travel.rasp.library.python.common23.models.fields import CodeCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager


@deconstructible
class StringTemplateValidator(object):
    def __init__(self, fields):
        self._fields = set(fields)

    def __call__(self, value):
        try:
            parsed_value = list(Formatter().parse(value))
        except ValueError as e:
            raise ValidationError(_(u"Ошибка в шаблоне {!r}").format(e))

        for _literal, field_name, _format, _conversion in parsed_value:
            if not (field_name is None or field_name in self._fields):
                raise ValidationError(
                    _(u"Неизвестное значение {!r}").format(field_name)
                )

    # Нужно, чтобы джанга не дублировала миграции
    def __eq__(self, other):
        return self._fields == other._fields


class SaleURLFormatter(Formatter):
    def format_field(self, value, format_spec):
        formatted = super(SaleURLFormatter, self).format_field(value, format_spec)

        return urlquote(formatted)


class BaseSupplier(models.Model):
    u""" Поименование рейсов при импорте """
    old_approved_codes = [
        u'airport_donetsk', u'airport_kharkov', u'airport_simferopol', u'arch_av',
        u'avto_siktivcar', u'a_linii', u'b2_msq', u'bb_travel', u'BS_Tushinskaya',
        u'chel_gortrans', u'dyc_ast', u'ekb_northav', u'IP_Bezuglov', u'IP_Stachanov',
        u'IP_Vildanova', u'Kostroma_avia', u'kvc_tula', u'maykop_av',
        u'middle_station_route_af', u'msk_subway', u'next_bus_tour',
        u'Nordexpress_Rybinsk', u'STK_Parus', u'Tarasovskoe_ATP', u'tis_from_hp', u'zd_syktyvkar',
    ]

    title = models.CharField(verbose_name=_(u'название организации поставщика'),
                             max_length=100, blank=False)
    code = CodeCharField(verbose_name=_(u'код организации поставщика'),
                         max_length=100, blank=False, unique=True, null=False,
                         validators=[RegexValidator(
                             r'(?u)^([0-9a-zA-Z-]*|' + u'|'.join(re.escape(oc) for oc in old_approved_codes) + r')$',
                             _('В коде поставщика могут учавствовать только '
                               'английские буквы цифры и знак минус'))])

    lmt = models.DateTimeField(verbose_name=_(u'дата последнего импорта'),
                               null=False, blank=False, auto_now_add=True,
                               help_text=_(u'используется только для ТИС'))
    filename = models.CharField(verbose_name=_(u'файл для импорта'),
                                max_length=100, null=True, blank=True)

    hide_in_filters = models.BooleanField(_(u"Не показывать в фильтрах"), default=False)

    exclude_from_external_api = models.BooleanField(_(u"Исключить из внешнего API"), default=False)

    can_buy_ru = models.BooleanField(
        _(u'Разрешить продажу билетов (ru)'), blank=False, null=False, default=False
    )

    can_buy_ua = models.BooleanField(
        _(u'Разрешить продажу билетов (ua)'), blank=False, null=False, default=False
    )

    can_buy_tr = models.BooleanField(
        _(u'Разрешить продажу билетов (tr)'), blank=False, null=False, default=False
    )

    logo = models.ImageField(
        _(u'Логотип'),
        storage=mds_s3_media_storage,
        blank=True, null=True, default=None,
        upload_to='data/supplier_logo'
    )

    sale_url_template = models.CharField(
        _(u'Шаблон ссылки'), max_length=1024, blank=True, null=False, default=u'',
        help_text=(u'Например, https://example.org/buy/?from={from_code}&to={to_code}'
                   u'&date={departure:%Y-%m-%d}&time={departure:%H:%M}'),
        validators=[
            StringTemplateValidator(fields=('from_code', 'to_code', 'departure'))
        ]
    )

    sale_start_days = models.IntegerField(
        _(u'Дни продажи'), blank=True, null=True,
        help_text=_(u'За сколько суток до отправления начинается продажа')
    )

    sale_stop_hours = models.IntegerField(
        _(u'Часы окончания продажи'), blank=True, null=True,
        help_text=_(u'За сколько часов до отправления заканчивается продажа')
    )

    displace_yabus = models.BooleanField(
        _(u'Замещать цены Яндекс.Автобусов'), null=False, default=False
    )

    objects = PrecachingManager(keys=['pk', 'code'])

    __codes_regexp = None

    def __unicode__(self):
        return self.title

    @classmethod
    def codes_regexp(cls):
        if not cls.__codes_regexp:
            cls.__codes_regexp = re.compile("(%s)" % "|".join([s.code for s in cls.objects.all()]))

        return cls.__codes_regexp

    def is_sale_enabled(self, national_version):
        return getattr(self, 'can_buy_{}'.format(national_version), False)

    def build_sale_url(self, *args, **kwargs):
        return SaleURLFormatter().format(self.sale_url_template, *args, **kwargs)

    class Meta(object):
        verbose_name = _(u'поставщик данных')
        verbose_name_plural = _(u'поставщики данных')
        app_label = 'www'
        abstract = True
