# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import itertools
import logging

from django.utils.functional import cached_property

from travel.rasp.library.python.common23.models.texts.i18n import translation


log = logging.getLogger(__name__)


class RouteLTitle(object):
    """
    Позволяет вычислять title объекта по последовательности точек (станций, городов, стран).

    В модели должен быть объявлен как RouteLTitle.L_field()
    """

    LANGUAGES = ['ru', 'tr', 'uk']

    prefetched_points = None

    def __init__(self, instance):
        self.instance = instance

    @cached_property
    def title_dict(self):
        return self.extract_title_dict(self.instance.title_common)

    @classmethod
    def get_base_language(cls, obj):
        return 'ru'

    @classmethod
    def tankerupload(cls, tanker, t_type_code):
        from travel.rasp.library.python.common23.models.core.schedule.rthread import RThread
        from travel.rasp.library.python.common23.models.core.schedule.route import Route

        keys = {}

        routes = Route.objects.filter(is_manual_title=True)
        threads = RThread.objects.filter(is_manual_title=True)
        if t_type_code != 'all':
            routes = routes.filter(t_type__code=t_type_code)
            threads = threads.filter(route__t_type__code=t_type_code)

        objects = itertools.chain(routes, threads)

        for obj in objects:
            key = obj.title

            translations = {}

            base_language = cls.get_base_language(obj)

            for lang in cls.LANGUAGES:
                if lang == base_language:
                    form = obj.title
                    author = None
                else:
                    form = getattr(obj, 'title_%s' % lang)
                    author = getattr(obj, 'title_%s_author' % lang)

                if form:
                    translations[lang] = {
                        "form": form,
                    }

                    if author:
                        translations[lang]["author"] = author

            keys[key] = {
                "info": {'is_plural': False},
                "translations": translations
            }

        tanker.upload('route_titles', keys, cls.LANGUAGES)

    @classmethod
    def tankerdownload(cls, tanker):
        from travel.rasp.library.python.common23.models.core.schedule.rthread import RThread
        from travel.rasp.library.python.common23.models.core.schedule.route import Route

        keys = tanker.download('route_titles', cls.LANGUAGES)

        objects = itertools.chain(Route.objects.filter(is_manual_title=True),
                                  RThread.objects.filter(is_manual_title=True))

        for obj in objects:
            key = obj.title

            base_language = cls.get_base_language(obj)

            for lang in cls.LANGUAGES:
                if lang == base_language:
                    continue

                if key in keys:
                    translation = keys[key]["translations"][lang]

                    form = translation["form"]

                    if form:
                        setattr(obj, 'title_%s' % lang, form)
                        setattr(obj, 'title_%s_author' % lang, translation["author"])

            obj.save()

    def __call__(self, lang=None, short=False, popular=False):
        from travel.rasp.library.python.common23.models.core.geo.title_generator import TitleGeneratorError, TitleGenerator

        instance = self.instance

        if lang is None:
            lang = translation.get_language()

        if getattr(instance, 'is_manual_title', False):
            title = getattr(instance, 'title_{}'.format(lang), instance.title)

            if title:
                return title

        base_language = self.get_base_language(instance)

        if self.title_dict and (lang != base_language or popular):
            try:
                return TitleGenerator.L_title(
                    self.title_dict, self.prefetched_points,
                    short=short, lang=lang, popular=popular
                )
            except TitleGeneratorError:
                log.exception(u'Ошибка генерации названия %r', self.title_dict)

        if short:
            return instance.title_short or instance.title

        return instance.title

    @classmethod
    def extract_title_dict(cls, title_common):
        if not title_common:
            return

        try:
            return json.loads(title_common)
        except ValueError:
            # Поддержка старого title_common
            return {
                'type': 'default',
                'title_parts': title_common.split(u'_')
            }

    @property
    def point_keys(self):
        from travel.rasp.library.python.common23.models.core.geo.title_generator import TitleGenerator

        if self.title_dict:
            return TitleGenerator.extract_point_keys(self.title_dict)

        return ()

    @classmethod
    def fetch(cls, fields):
        from travel.rasp.library.python.common23.models.core.geo.point import Point

        point_keys = set()

        for field in fields:
            point_keys.update(field.point_keys)

        points = Point.in_bulk(point_keys)

        for field in fields:
            field.prefetched_points = points

    class L_field(object):

        def contribute_to_class(self, cls, name):
            self.name = name
            setattr(cls, name, self)

        def __get__(self, instance, _instance_type=None):
            if instance is None:
                return self

            field = RouteLTitle(instance)

            # кэшируем поле в инстансе
            setattr(instance, self.name, field)

            return field
