# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import pytz
import warnings
from datetime import datetime, timedelta

from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.date.date import smart_localize
from travel.rasp.library.python.common23.models.texts.platform_translation import PlatformTranslation
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning


class RTStationLogicMixin(object):
    __slots__ = []

    def L_platform(self, lang=None):
        return PlatformTranslation.get_translation(self.platform, lang)

    def get_pytz(self):
        return pytz.timezone(self.time_zone)

    @property
    def pytz(self):
        return self.get_pytz()

    def get_arrival_mask_shift(self, thread_start_time):
        if self.tz_arrival is None:
            return None

        return (self.tz_arrival + thread_start_time.hour * 60 + thread_start_time.minute) // 1440

    def get_departure_mask_shift(self, thread_start_time):
        if self.tz_departure is None:
            return None

        return (self.tz_departure + thread_start_time.hour * 60 + thread_start_time.minute) // 1440

    def get_arrival_dt(self, naive_start_dt, out_tz=None):
        """
        @return: Aware Arrival datetime
        """
        if self.tz_arrival is None:
            return

        arrival_dt = smart_localize(naive_start_dt + timedelta(minutes=self.tz_arrival), self.pytz)

        if out_tz:
            return arrival_dt.astimezone(out_tz)
        else:
            return arrival_dt

    def get_arrival_loc_dt(self, naive_start_dt):
        return self.get_arrival_dt(naive_start_dt, self.station.pytz)

    def get_loc_arrival_dt(self, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_arrival_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_arrival_loc_dt(naive_start_dt)

    def get_departure_dt(self, naive_start_dt, out_tz=None):
        """
        @return: Aware departure datetime
        """
        if self.tz_departure is None:
            return

        departure_dt = smart_localize(naive_start_dt + timedelta(minutes=self.tz_departure), self.pytz)

        if out_tz:
            return departure_dt.astimezone(out_tz)
        else:
            return departure_dt

    def get_departure_loc_dt(self, naive_start_dt):
        return self.get_departure_dt(naive_start_dt, self.station.pytz)

    def get_loc_departure_dt(self, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_departure_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_departure_loc_dt(naive_start_dt)

    def get_event_dt(self, event, naive_start_dt, out_tz=None):
        return getattr(self, 'get_{}_dt'.format(event))(naive_start_dt, out_tz)

    def get_event_loc_dt(self, event, naive_start_dt):
        return self.get_event_dt(event, naive_start_dt, self.station.pytz)

    def get_loc_event_dt(self, event, naive_start_dt):
        warnings.warn('[2015-07-21] Use get_event_loc_dt', RaspDeprecationWarning, stacklevel=2)

        return self.get_event_loc_dt(event, naive_start_dt)

    def calc_days_shift(self, event='departure', start_date=None, event_tz=None):
        """
        Вычисляет сдвиг в днях
            от старта нитки
                во временной зоне старта нитки
            до отправления со станции (прибытия на станцию) self
                во временной зоне event_tz (по-умолчанию во временной зоне self.station)
        если
            известен день старта нитки start_date во временной зоне нитки
            по-умолчанию берем дату во временной зоне нитки на сейчас
        """

        return self.calc_days_timedelta(event, start_date, event_tz).days

    def calc_days_timedelta(self, event='departure', start_date=None, event_tz=None):
        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if start_date is None:
            start_date = environment.now_aware().astimezone(self.thread.pytz).date()

        naive_start_dt = datetime.combine(start_date, self.thread.tz_start_time)

        event_dt = self.get_event_dt(event, naive_start_dt, out_tz=event_tz)

        return event_dt.date() - naive_start_dt.date()

    def calc_days_shift_by_event_date(self, event='departure', event_date=None, event_tz=None):
        """
        Вычисляет сдвиг в днях
            от старта нитки
                во временной зоне старта нитки
            до отправления со станции (прибытия на станцию) self
                во временной зоне event_tz (по-умолчанию во временной зоне self.station)
        если
            известна дата отправления(прибытия)
        """

        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if event_date is None:
            event_date = environment.now_aware().astimezone(event_tz).date()

        start_date = self.calc_thread_start_date(event, event_date, event_tz)

        return self.calc_days_shift(event, start_date, event_tz)

    def calc_thread_start_date(self, event='departure', event_date=None, event_tz=None):
        """
        Вычисляем время старта нитки так,
        чтобы дата отправления(прибытия) совпала с нашей
        """

        if event not in ['departure', 'arrival']:
            raise ValueError("Event must be 'departure' or 'arrival'")

        if event_tz is None:
            event_tz = self.station.pytz

        if event_date is None:
            event_date = environment.now_aware().astimezone(event_tz).date()

        try_days_timedelta = self.calc_days_timedelta(event, start_date=event_date, event_tz=event_tz)

        try_naive_start_dt = datetime.combine(event_date, self.thread.tz_start_time)

        try_start_date = try_naive_start_dt.date() - try_days_timedelta

        # Сдвиг даты старта нитки по thread.pytz от даты отправления со станции по from_date_tz,
        # при переходе на летнее(зимнее) время может отличатся на ± 1 день

        naive_start_dt = datetime.combine(try_start_date, self.thread.tz_start_time)
        try_from_date = self.get_event_dt(event, naive_start_dt, out_tz=event_tz).date()

        if try_from_date == event_date:
            return try_start_date

        elif try_from_date < event_date:
            return try_start_date + timedelta(1)

        else:
            return try_start_date - timedelta(1)

    def is_no_stop(self):
        """
        Проверяем является ли станция безостановочной.
        """
        return bool(self.tz_arrival and self.tz_departure and self.tz_arrival == self.tz_departure)
