# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import warnings
from datetime import timedelta
from pytils.dt import ru_strftime

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.date.date import NBSP, DateTimeFormatter
from travel.rasp.library.python.common23.models.fields import CodeCharField, TrimmedCharField
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning
from travel.rasp.library.python.common23.xgettext.i18n import gettext


class TrainSchedulePlan(models.Model):
    """
    График поездов и электричек
    """

    title = TrimmedCharField(verbose_name=_(u"название граффика"),
                             default=None, blank=True, null=True,
                             max_length=255)
    start_date = models.DateField(verbose_name=_(u"дата начала(включительно)"),
                                  null=False, blank=False)
    end_date = models.DateField(verbose_name=_(u"дата окончания(включительно)"),
                                null=False, blank=False)
    code = CodeCharField(verbose_name=_(u"код"), max_length=50,
                         null=False, blank=False, unique=True)

    APPENDIX_FROM = 'from'
    APPENDIX_TO = 'to'
    APPENDIX_CHOICES = (
        (APPENDIX_FROM, _(u'Использовать "c"')),
        (APPENDIX_TO, _(u'Использовать "по"')),
    )

    appendix_type = models.CharField(_(u'селектор с/по'), max_length=4,
                                     choices=APPENDIX_CHOICES, default=APPENDIX_FROM,
                                     null=False, blank=False)

    @classmethod
    def fill_thread_plans(cls, threads):
        plans_ids = set(t.schedule_plan_id for t in threads)

        plans_by_id = TrainSchedulePlan.objects.in_bulk(list(plans_ids))

        for t in threads:
            t.schedule_plan = plans_by_id.get(t.schedule_plan_id)

        return list(plans_by_id.values())

    @classmethod
    def add_to_threads(cls, threads, today):
        """
        @Deprecated
        На каждый thread устанавливается соответствующий schedule_plan.
        Возвращается текущий и следующий планы (они одинаковы для всех ниток).
        """
        plans = cls.fill_thread_plans(threads)

        if plans:
            plans_qs = cls.objects.filter(id__in=[p.id for p in plans])
            current_plan, next_plan = cls.get_current_and_next(today, qs=plans_qs)
            return current_plan, next_plan
        else:
            return None, None

    @classmethod
    def get_current_and_next(cls, today, qs=None):
        """
        Отображаемые графики зависят от сегоднящнего дня.
        Не от дня отображаемого расписания!
        """
        if qs is None:
            qs = cls.objects

        current = next_plan = None

        # Расчитываем на то, что планы не пересекаются и идут по порядку
        plans = list(qs.filter(end_date__gte=today).order_by('start_date')[:2])

        if not plans:
            return current, next_plan

        if plans[0].start_date <= today:
            current = plans.pop(0)
            current.name = None

        if plans:
            next_plan = plans[0]
            start_date_str = DateTimeFormatter(next_plan.start_date, lang='ru').L().replace(' ', NBSP)
            next_plan.name = u'с&nbsp;{}'.format(start_date_str)

        if current and (next_plan or (current.end_date - today).days < 7):
            end_date_str = DateTimeFormatter(current.end_date, lang='ru').L().replace(' ', NBSP)
            current.name = u'по&nbsp;{}'.format(end_date_str)

        return current, next_plan

    def is_current(self, today):
        return self.start_date <= today <= self.end_date

    class Meta(object):
        verbose_name = _(u"график поездов и электричек")
        verbose_name_plural = _(u"графики поездов и электричек")
        app_label = 'www'
        db_table = 'www_trainscheduleplan'

    def get_appendix(self, thread_start_date, next_plan):
        """
        @Deprecated
        Первоначальный таск - RASP-6846 ?
        """

        warnings.warn("Use TrainSchedulePlan.get_L_appendix() or L_appendix()",
                      RaspDeprecationWarning, stacklevel=2)

        if self == next_plan:
            return ru_strftime(u"c %d %B", self.start_date, inflected=True)

        elif next_plan or self.end_date - thread_start_date < timedelta(days=20):
            return ru_strftime(u"по %d %B", self.end_date, inflected=True)

    def L_appendix(self):
        if self.appendix_type == self.APPENDIX_FROM:
            return gettext(u'с {human_date}').format(human_date=DateTimeFormatter(self.start_date).L())

        else:
            return gettext(u'по {human_date}').format(human_date=DateTimeFormatter(self.end_date).L())

    def get_L_appendix(self, thread_start_date, next_plan):
        """
        TODO: разобраться и сделать правильную логику работы
        1. Логика работы - взята старая, чтобы ничего не поломать
        2. Вообще странно, что передается thread_start_date
        3. Сейчас по факту в thread_start_date передается first_run или локализованый today
        """
        if self == next_plan:
            return self.L_appendix()

        elif next_plan or thread_start_date + timedelta(days=20) > self.end_date:
            return self.L_appendix()

    @classmethod
    def get_schedule_end_period(cls, today=None):
        today = today or environment.today()

        end_dates = [
            tsp.end_date for tsp in
            cls.objects.filter(end_date__gte=today, end_date__isnull=False)
        ]

        if end_dates:
            return min(end_dates)
