# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime

from django.conf import settings
from django.core.cache import cache
from django.utils.http import urlencode
from lxml import etree
from pybreaker import CircuitBreaker

from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.utils.http import urlopen
from travel.library.python.tracing.instrumentation import traced_function

log = logging.getLogger(__name__)

define_setting('CURRENCY_CONVERTER_BREAKER_PARAMS', default={'fail_max': 3, 'reset_timeout': 60})

currency_converter_breaker = CircuitBreaker(**settings.CURRENCY_CONVERTER_BREAKER_PARAMS)


CURRENCY_RATES_GEOID = settings.MOSCOW_GEO_ID
CURRENCY_RATES_CACHE_KEY = 'currency_rates'

NATIONAL_CURRENCY_RATES_GEOID = {
    'ru': settings.MOSCOW_GEO_ID,
    'ua': settings.KIEV_GEO_ID,
    'tr': settings.ISTANBUL_GEO_ID,
}

CURRENCY_RATES_CACHE_TIMEOUT = 60 * 60 * 24


class ConverterCache(object):
    def __init__(self, rates, actual_datetime, national_version):
        self.rates = rates
        self.actual_datetime = actual_datetime
        self.national_version = national_version

    @classmethod
    def load(cls, national_version='ru'):
        return cache.get(cls._get_key(national_version))

    @classmethod
    def build(cls, national_version='ru'):
        from travel.rasp.library.python.common23.models.currency.currency import Currency
        currencies = Currency.objects.all()
        src, rates = Currency.fetch_rates(
            currencies,
            cls.get_currency_geoid(national_version),
            base_currency=cls.get_national_currency(national_version)
        )

        if src is None:
            raise RuntimeError('no data from currency converter')

        return cls(rates, datetime.now(), national_version)

    def save(self):
        cache.set(
            key=self._get_key(self.national_version),
            value=self,
            timeout=CURRENCY_RATES_CACHE_TIMEOUT
        )

    @classmethod
    def _get_key(cls, national_version):
        return '/yandex/rasp/%s/%s' % (
            CURRENCY_RATES_CACHE_KEY, national_version
        )

    @classmethod
    def get_currency_geoid(cls, national_version):
        return NATIONAL_CURRENCY_RATES_GEOID[national_version]

    @classmethod
    def get_national_currency(cls, national_version):
        return settings.NATIONAL_CURRENCIES[national_version]


AVIA_CURRENCY = ['AUD', 'BGN', 'BRL', 'BYR', 'CAD', 'CHF', 'CNY', 'CZK', 'DKK', 'EUR', 'GBP', 'HKD', 'HRK', 'HUF',
                 'IDR', 'ILS', 'INR', 'JPY', 'KRW', 'KZT', 'MXN', 'MYR', 'NOK', 'NZD', 'PHP', 'PLN', 'RON', 'RUB',
                 'RUR', 'SEK', 'SGD', 'THB', 'TRY', 'UAH', 'USD', 'UZS', 'ZAR']


def fetch_currency_rates(currencies, geoid=settings.MOSCOW_GEO_ID, base_currency='RUR'):
    """ https://wiki.yandex-team.ru/VladimirSavenkov/ConvRasp/ """
    if not settings.CURRENCY_RATES_URL:
        return None, {}

    params = {
        'from': base_currency,
        'to': currencies,
        'geoid': geoid,
    }

    @currency_converter_breaker
    @traced_function
    def get_response():
        url = settings.CURRENCY_RATES_URL + '?' + urlencode(params, doseq=True)
        tree = etree.parse(urlopen(url, timeout=settings.CURRENCY_RATES_TIMEOUT))
        status = tree.xpath("/response/status")[0].text
        if status != 'ok':
            return None, {}

        src = tree.xpath("/response/data/src")[0].text
        rates = dict((to.attrib['id'], float(to.attrib['value'])) for to in tree.xpath("/response/data/rates/to"))
        return src, rates

    try:
        return get_response()
    except Exception:
        log.exception('currency rates fetching exception:')

    return None, {}
