# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from functools import total_ordering

from travel.rasp.library.python.common23.models.currency.currency import Currency


@total_ordering
class Price(object):
    def __init__(self, value, currency=None):
        self.value = value
        self.currency = currency or Currency.BASE_CURRENCY

        self.base_value = None

        if self.currency == Currency.BASE_CURRENCY:
            self.base_value = value

    def rebase(self, rates):
        if self.currency == Currency.BASE_CURRENCY:
            return

        rate = rates.get(self.currency)

        if rate is not None:
            self.base_value = self.value * rate

    def __json__(self):
        v = {
            'value': self.value,
            'currency': self.currency,
        }

        if self.base_value:
            v['baseValue'] = self.base_value

        return v

    @classmethod
    def parse(cls, s):
        if not s:
            raise ValueError

        parts = s.split()

        value = float(parts[0])

        currency = None

        if len(parts) > 1:
            currency = parts[1]

        return cls(value, currency)

    def __repr__(self):
        return '<Price base_value=%r currency=%r value=%r>' % (self.base_value, self.currency, self.value)

    def __str__(self):
        return "%s %s" % (self.value, self.currency)

    @property
    def sort_value(self):
        return [self.base_value, self.value]

    def __lt__(self, other):
        if not isinstance(other, Price):
            other = Price(other)

        return self.sort_value < other.sort_value

    def __eq__(self, other):
        if not isinstance(other, Price):
            other = Price(other)

        return self.sort_value == other.sort_value

    def __mul__(self, other):
        return Price(self.value * other, self.currency)

    def __bool__(self):
        return int(self.value)

    __nonzero__ = __bool__

    def __add__(self, other):
        if other.currency != self.currency:
            raise ValueError("Can't sum prices with different currencies %s and %s" % (self.currency, other.currency))

        return Price(self.value + other.value, self.currency)
