# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import six
from builtins import bytes

from django.core.urlresolvers import reverse
from django.db import models
from django.utils.safestring import SafeText

from travel.rasp.library.python.common23.models.fields.forms import (
    RegExpFormField, TextFileFormField, TextFileWidget, DatabaseFileFormField, DatabaseFileWidget, ThreadCalendarWidget
)
from travel.rasp.library.python.common23.utils.memory_file import MemoryFile


class TrimmedCharField(models.CharField):
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            if isinstance(value, six.string_types):
                value = value.strip()
        setattr(instance, self.attname, value)
        return value


class TrimmedTextField(models.TextField):
    def pre_save(self, instance, add):
        value = models.TextField.pre_save(self, instance, add)

        if value is not None:
            if isinstance(value, six.string_types):
                value = value.strip()
        setattr(instance, self.attname, value)
        return value


class RegExpField(models.CharField):
    def __init__(self, *args, **kwargs):
        super(RegExpField, self).__init__(*args, **kwargs)

    def formfield(self, **kwargs):
        defaults = {'form_class': RegExpFormField}
        defaults.update(kwargs)
        return super(RegExpField, self).formfield(**defaults)


class RegExpTextField(models.TextField):
    def __init__(self, *args, **kwargs):
        super(RegExpTextField, self).__init__(*args, **kwargs)

    def formfield(self, **kwargs):
        defaults = {'form_class': RegExpFormField}
        defaults.update(kwargs)
        return super(RegExpTextField, self).formfield(**defaults)


class TextFileDescriptor(object):
    def __init__(self, field):
        self.field = field

    def __get__(self, instance=None, owner=None):
        if instance is None:
            raise AttributeError(
                "The '%s' attribute can only be accessed from %s instances."
                % (self.field.name, owner.__name__))

        if instance.__dict__[self.field.name] is None:
            return None

        f = SafeText(instance.__dict__[self.field.name])

        f.encoding = self.field.encoding
        f.url = '/admin/show_file/%s/%s/%s/%s' % (instance._meta.app_label,
                                                  instance.__class__.__name__,
                                                  self.field.name,
                                                  instance.pk)

        f.name = getattr(instance, self.field.name + '_name', None)

        return f

    def __set__(self, instance, value):
        instance.__dict__[self.field.name] = value


class TextFileField(models.Field):
    descriptor_class = TextFileDescriptor

    def __init__(self, *args, **kwargs):

        self.is_xml = kwargs.pop('is_xml', False)
        self.encoding = kwargs.pop('encoding', 'utf8')

        super(TextFileField, self).__init__(*args, **kwargs)

    def get_internal_type(self):
        return "TextField"

    def contribute_to_class(self, cls, name):
        super(TextFileField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, self.descriptor_class(self))

    def save_form_data(self, instance, data):
        setattr(instance, self.name, data)

        if hasattr(data, 'read'):
            if not getattr(instance, self.name + '_name'):
                setattr(instance, self.name + '_name', data.name)

            setattr(instance, self.name, data.read().decode(self.encoding))

    def formfield(self, **kwargs):
        defaults = {
            'form_class': TextFileFormField,
            'max_length': self.max_length,
            'encoding': self.encoding,
            'widget': TextFileWidget,
            'is_xml': self.is_xml}
        # If a file has been provided previously, then the form doesn't require
        # that a new file is provided this time.
        # The code to mark the form field as not required is used by
        # form_for_instance, but can probably be removed once form_for_instance
        # is gone. ModelForm uses a different method to check for an existing file.
        if 'initial' in kwargs:
            defaults['required'] = False
        defaults.update(kwargs)
        return super(TextFileField, self).formfield(**defaults)


class DatabaseFileDescriptor(object):
    def __init__(self, field):
        self.field = field

    def __get__(self, instance=None, owner=None):
        if instance is None:
            raise AttributeError(
                "The '%s' attribute can only be accessed from %s instances."
                % (self.field.name, owner.__name__))

        if instance.__dict__[self.field.name] is None:
            return None

        memory_file = instance.__dict__[self.field.name]

        if instance.id:
            memory_file.download_url = reverse('admin_download_db_file',
                                               kwargs={'app_label': instance._meta.app_label,
                                                       'model_name': instance.__class__.__name__,
                                                       'pk': instance.id,
                                                       'field_name': self.field.name})

            memory_file.delete_url = reverse('admin_delete_db_file',
                                             kwargs={'app_label': instance._meta.app_label,
                                                     'model_name': instance.__class__.__name__,
                                                     'pk': instance.id,
                                                     'field_name': self.field.name})

        return memory_file

    def __set__(self, instance, value):
        instance.__dict__[self.field.name] = self.field.to_python(value)


class DatabaseFileField(models.Field):
    descriptor_class = DatabaseFileDescriptor

    def contribute_to_class(self, cls, name):
        super(DatabaseFileField, self).contribute_to_class(cls, name)
        setattr(cls, self.name, self.descriptor_class(self))

    def __init__(self, *args, **kwargs):
        self.content_type = kwargs.pop('content_type', 'application/octet-stream')
        super(DatabaseFileField, self).__init__(*args, **kwargs)

    def db_type(self, connection):
        return 'LONGBLOB'

    def to_python(self, value):
        if isinstance(value, MemoryFile):
            return value

        if not value:
            return None

        if isinstance(value, bytes):
            return MemoryFile.from_binary_string(value)
        elif hasattr(value, 'name') and hasattr(value, 'read'):
            return MemoryFile.from_file_like_object_string(value, self.content_type)
        else:
            raise TypeError("Conversion from %s type not supported" % type(value))

    def get_prep_value(self, value):
        if isinstance(value, MemoryFile):
            return value.to_binary_string()
        else:
            return value

    def get_prep_lookup(self, lookup_type, value):
        # We only handle 'exact'. All others are errors.
        if lookup_type == 'exact':
            return self.get_prep_value(value)
        raise TypeError('Lookup type %r not supported.' % lookup_type)

    def formfield(self, **kwargs):
        defaults = {
            'form_class': DatabaseFileFormField,
            'widget': DatabaseFileWidget,
            'content_type': self.content_type
        }
        # If a file has been provided previously, then the form doesn't require
        # that a new file is provided this time.
        # The code to mark the form field as not required is used by
        # form_for_instance, but can probably be removed once form_for_instance
        # is gone. ModelForm uses a different method to check for an existing file.
        if 'initial' in kwargs:
            defaults['required'] = False
        defaults.update(kwargs)
        return super(DatabaseFileField, self).formfield(**defaults)


class CodeCharField(models.CharField):
    u"""
    Обращает пустую строку в None, чтобы было удобнее обращаться с unique=True
    """
    def pre_save(self, instance, add):
        value = models.CharField.pre_save(self, instance, add)

        if value is not None:
            value = value.strip() or None
            setattr(instance, self.attname, value)
            return value


class ThreadCalendarField(models.Field):
    """ Класс для поля, хранящего дни хождения нитки """

    def formfield(self, **kwargs):
        defaults = {'widget': ThreadCalendarWidget}
        defaults.update(kwargs)
        return super(ThreadCalendarField, self).formfield(**defaults)

    def get_internal_type(self):
        return 'TextField'


class UUID64Field(models.BigIntegerField):
    pass
