# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import range
from builtins import str
import calendar
import re
from datetime import date
from itertools import groupby
from operator import attrgetter

from django import forms
from django.conf import settings
from django.template.loader import render_to_string
from django.utils.encoding import force_text
from django.utils.safestring import mark_safe
from xml.etree import ElementTree
from xml.parsers.expat import ExpatError

from travel.rasp.library.python.common23.date.date import uni_strftime
from travel.rasp.library.python.common23.date.run_mask import RunMask
from travel.rasp.library.python.common23.utils.memory_file import MemoryFile


class RegExpFormField(forms.CharField):
    def clean(self, value):
        value = super(RegExpFormField, self).clean(value)
        try:
            value = value.strip()
            re.compile(value, re.U)
        except re.error as e:
            raise forms.ValidationError(u'Ошибка в регулярном выражении: {}'.format(force_text(e)))
        return value


class TextFileWidget(forms.FileInput):
    """
    A FileField Widget that shows its current value if it has one.
    """
    def render(self, name, value, attrs=None, renderer=None):
        output = []
        if value and hasattr(value, "url"):
            output.append('%s <a target="_blank" href="%s">%s</a> <br />' %
                          (u'Скачать:', value.url, value.name))
        output.append(super(TextFileWidget, self).render(name, value, attrs, renderer=renderer))
        return mark_safe(u''.join(output))


class TextFileFormField(forms.FileField):
    def __init__(self, encoding='utf8', is_xml=False, *args, **kwargs):
        self.encoding = encoding
        self.is_xml = is_xml
        super(TextFileFormField, self).__init__(*args, **kwargs)

    def clean(self, value, initial=None):
        u"""Для FileField в clean передается два аргумента вместо одного"""
        if value is None and initial:
            return initial
        value = super(TextFileFormField, self).clean(value)

        if value is None:
            return None

        try:
            data = value.read().decode(self.encoding)
            value.seek(0)
        except UnicodeDecodeError:
            raise forms.ValidationError(
                u'Кодировка файла не совпадает с ожидаемой кодировкой: {}'.format(
                    force_text(self.encoding)
                ))
        if self.is_xml:
            try:
                ElementTree.fromstring(data.encode(self.encoding))
            except (ExpatError, ElementTree.ParseError) as e:
                raise forms.ValidationError(u'Не валидный xml: {}'.format(force_text(e)))
        return value


class DatabaseFileWidget(forms.FileInput):
    """
    A FileField Widget that shows its current value if it has one.
    """
    def render(self, name, value, attrs=None, renderer=None):

        output = []
        if value and hasattr(value, "download_url"):
            output.append(
                u"""
<div style="float: left">
%(name)s <a target="_blank" href="%(donwload_url)s">Скачать</a> |
<a href="%(delete_url)s" onclick="return confirm('Вы действительно хотите удалить файл?');">Удалить</a><br />
                """.strip() % ({
                    'donwload_url': value.download_url,
                    'name': value.name,
                    'delete_url': value.delete_url,
                })
            )
        output.append(super(DatabaseFileWidget, self).render(name, value, attrs, renderer=renderer))

        if value and hasattr(value, "download_url"):
            output.append(u"</div>")

        return mark_safe(u''.join(output))


class DatabaseFileFormField(forms.FileField):
    def __init__(self, content_type='application/octet-stream', *args, **kwargs):
        self.content_type = content_type
        super(DatabaseFileFormField, self).__init__(*args, **kwargs)

    def clean(self, value, initial=None):
        u"""Для FileField в clean передается два аргумента вместо одного"""
        if value is None and initial:
            return initial

        value = super(DatabaseFileFormField, self).clean(value)

        if value is None:
            return None

        value = MemoryFile(value.name, self.content_type, value.read())

        return value


class ThreadCalendarWidget(forms.Widget):
    class Media(object):
        css = {'all': [settings.STATIC_URL + 'rasp/calendar_widget.css']}
        js = [settings.STATIC_URL + 'rasp/calendar_widget.js']

    @staticmethod
    def make_calendar(daily_data_by_month):
        """
        Возвращает календарь (год, разбитый на месяцы и недели)
        с данными, подключенными к дням, в виде списка месяцев.
        Месяц - это список недель. Каждая неделя - список пар
        (day, day_data), где day и day_data равны None, если
        эти дни за пределами месяца.

        Год подразумевается текущим, если передан как None
        daily_data_by_month - список списков, по списку на месяц.
        """

        months = []

        for first_month_day, month_data in daily_data_by_month:
            month = first_month_day.month
            weeks = calendar.monthcalendar(first_month_day.year, month)

            for week in weeks:
                # i нужен, чтобы модифицировать список i на месте (экономим память)
                for i, day in enumerate(week):
                    # Ноль означает, что день за пределами месяца
                    if day:
                        # Пробуем получить данные
                        attrs = {
                            'holiday': i > 4,
                        }

                        if month_data is not None:
                            day_data = month_data[day - 1]
                        else:
                            day_data = None

                        week[i] = (date(first_month_day.year, month, day), day_data, attrs)

                    else:
                        week[i] = (None, None, None)

            months.append({
                'number': first_month_day.month,
                'name': uni_strftime(u'%B', first_month_day),
                'weeks': weeks,
                'first_day': first_month_day,
                'data': month_data,
            })

        return months

    def render(self, name, value, attrs=None, renderer=None):
        if not value:
            value = RunMask.EMPTY_YEAR_DAYS

        mask = RunMask(value, date.today())

        # Нормализуем value (RASP-4298)
        value = str(RunMask(days=mask.dates()))

        data = []

        for _, dates in groupby(mask.all_days, key=attrgetter('month')):
            dates = list(dates)
            first = dates[0].replace(day=1)

            month_data = [None] * 31

            for d in dates:
                month_data[d.day - 1] = mask[d] and '1' or '0'

            data.append((first, month_data))

        pad = (data[0][0].month - 1) % 3

        if pad > 0:
            first_day = data[0][0]
            data = [(date(first_day.year, first_day.month - pad + i, 1), None) for i in range(pad)] + data

        pad_right = 2 - (data[-1][0].month - 1) % 3

        if pad_right > 0:
            first_day = data[-1][0]
            data = data + [(date(first_day.year, first_day.month + i + 1, 1), None) for i in range(pad_right)]

        months = self.make_calendar(data)

        return render_to_string('admin/block_calendar.html', {'months': months, 'days': value, 'name': name,
                                                              'input_id': attrs['id']})
