# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from builtins import map
from builtins import range
from functools import reduce
"""
    Пример конфигурации:

    # количество попыток кеширования в случае ошибок. по-умолчанию количество не ограничено
    PRECACHE_SETUP_RETRIES = 10

    # список путей до объектов с методом using_precache, возвращающим контекстный менеджер для заполнения кеша
    PRECACHE_MANAGERS = (
        'package.module:ClassName.manager',
        ...
    )
"""

import itertools
import logging
import os
import random
import resource
import signal
import warnings
from importlib import import_module
from time import (
    sleep as time_sleep,  # import sleep нужен для патчинга в тестах так как патч модуля time ломает клиента mongodb
    time as time_time
)

from contextlib2 import ExitStack, contextmanager
from django.conf import settings
from django.db import connections
from raven.contrib.django.raven_compat.models import client

from travel.rasp.library.python.common23.db.switcher import switcher
from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning

log = logging.getLogger(__name__)

define_setting('WAIT_BEFORE_RECACHE', default=60 * 5)


def _import_attribute(path):
    module_name, attr_path = path.split(':')
    return reduce(getattr, attr_path.split('.'), import_module(module_name))


def _get_memory_usage():
    return resource.getrusage(resource.RUSAGE_SELF).ru_maxrss


@contextmanager
def _manager_debug_log(manager_path):
    start_time = time_time()
    start_memory = _get_memory_usage()
    yield
    log.debug(
        '%s used %s KB in %.3f seconds',
        manager_path, _get_memory_usage() - start_memory, time_time() - start_time
    )


class MethodsPrecacheBuilder(object):
    def __init__(self, method_paths):
        self.methods = list(map(_import_attribute, method_paths))

    def __call__(self):
        for method in self.methods:
            method()


class ManagersPrecacheBuilder(object):
    def __init__(self, manager_paths):
        self.manager_paths = manager_paths
        self.precached_stack = ExitStack()

    def __call__(self):
        self.precached_stack.close()

        with ExitStack() as precache_stack:
            for manager_path in self.manager_paths:
                manager = _import_attribute(manager_path)
                with _manager_debug_log(manager_path):
                    precache_stack.enter_context(manager.using_precache())

            self.precached_stack = precache_stack.pop_all()


def _make_precache_builder():
    methods_paths = getattr(settings, 'PRECACHE', None)
    managers_paths = getattr(settings, 'PRECACHE_MANAGERS', None)

    if not methods_paths and not managers_paths:
        return None

    if methods_paths:
        warnings.warn('[2017-11-13] use PRECACHE_MANAGERS setting with managers', RaspDeprecationWarning)
        return MethodsPrecacheBuilder(methods_paths)

    return ManagersPrecacheBuilder(managers_paths)


precache = _make_precache_builder()


RECACHE_FUNC_UID = 'travel.rasp.library.python.common23.precache.backend.recache'


@contextmanager
def _sync_and_disconnect():
    switcher.sync_db()
    try:
        yield
    finally:
        for connection in connections.all():
            connection.close()


def setup_precache(logger):
    if precache is None:
        logger.info('Precache is disabled')
        return

    retries = getattr(settings, 'PRECACHE_SETUP_RETRIES', None)
    tries_counter = itertools.repeat(True) if retries is None else reversed(list(range(retries)))

    switcher.data_updated.disconnect(dispatch_uid=RECACHE_FUNC_UID)

    for tries_left in tries_counter:
        logger.info('Precaching...')
        try:
            with _sync_and_disconnect():
                precache()
        except Exception:
            client.captureException()

            if not tries_left:
                raise

            logger.info('Unhandled exception in precache:', exc_info=True)
            logger.info('Retrying in 30 seconds')
            time_sleep(30)
        else:
            break

    server_pid = os.getpid()

    def recache(**kwargs):
        # https://st.yandex-team.ru/RASPFRONT-9300
        wait_time = random.randint(0, int(settings.WAIT_BEFORE_RECACHE))
        logger.info('Data update detected. Waiting for {}s before recache'.format(wait_time))
        time_sleep(wait_time)

        os.kill(server_pid, signal.SIGHUP)

    switcher.data_updated.connect(recache, weak=False, dispatch_uid=RECACHE_FUNC_UID)
