# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import range
import copy
import warnings
from collections import defaultdict
from contextlib import contextmanager
from operator import attrgetter

import six
from django.conf import settings
from django.db.models import Manager, Field
from django.db.models.lookups import BuiltinLookup
from django.db.models.query import QuerySet, ModelIterable

from travel.rasp.library.python.common23.models.utils import model_iter
from travel.rasp.library.python.common23.settings.utils import define_setting
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning

from .proxy_slots import get_slots_proxyficator

define_setting('PRECACHE_ENABLE_FALLBACK_BY_DEFAULT', default=True)


class CacheIsMissingError(Exception):
    pass


class PrecachedModelIterable(ModelIterable):
    def __iter__(self):
        precached_result_generator = self.queryset._get_precached_result_generator()

        if precached_result_generator:
            try:
                for obj in precached_result_generator:
                    yield obj
                return
            except CacheIsMissingError:
                if not (self.queryset.use_get_fallback if self.queryset.use_get_fallback is not None else
                        settings.PRECACHE_ENABLE_FALLBACK_BY_DEFAULT):
                    return

        for obj in super(PrecachedModelIterable, self).__iter__():
            yield obj


class BasePrecachingQuerySet(QuerySet):
    manager = None
    use_get_fallback = False

    def __init__(self, *args, **kwargs):
        super(BasePrecachingQuerySet, self).__init__(*args, **kwargs)

        self._iterable_class = PrecachedModelIterable

    def _get_precached_result_generator(self):
        if not self.manager.precached:
            return

        if self.query.select_related:
            return

        if self._prefetch_related_lookups:
            return

        if not self.query.where.children and not self.query.order_by:
            return self._unordered_all_generator()

        if len(self.query.where.children) != 1:
            return

        lookup = self.query.where.children[0]
        if not isinstance(lookup, BuiltinLookup):
            return

        is_the_same_table = lookup.lhs.alias == self.model._meta.db_table
        if not is_the_same_table:
            return

        lookup_name = lookup.lookup_name
        if lookup_name not in ('in', 'exact', 'iexact'):
            return

        lookup_field = isinstance(lookup.lhs.target, Field) and lookup.lhs.target

        if not lookup_field:
            return

        if lookup_field.primary_key:
            field_key = 'pk'
        else:
            field_key = lookup_field.name

        exact_cache_name = '{}_{}'.format(field_key, 'exact')
        iexact_cache_name = '{}_{}'.format(field_key, 'iexact')
        lookup_value = lookup.rhs

        model_caches = self.manager.precache_caches

        if lookup_name == 'in' and exact_cache_name in model_caches:
            return self._get_in_generator(exact_cache_name, lookup_value)
        elif lookup_name == 'exact':
            if exact_cache_name in model_caches:
                return self._get_exact_generator(exact_cache_name, lookup_value)
            # Делаем фоллбек на iexact, чтобы поддержать старое поведение с get(iata='AAA') ~ get(iata='BBB')
            elif iexact_cache_name in model_caches:
                warnings.warn('[2016-03-28] Do Not Use This Hack', RaspDeprecationWarning, stacklevel=7)
                return self._get_iexact_generator(iexact_cache_name, lookup_value)

        elif lookup_name == 'iexact' and iexact_cache_name in model_caches:
            return self._get_iexact_generator(iexact_cache_name, lookup_value)
        else:
            return

    def _unordered_all_generator(self):
        for objs in list(self.manager.precache_caches['pk_exact'].values())[self.query.low_mark:self.query.high_mark]:
            yield objs[0]

    def _get_exact_generator(self, cache_name, value):
        try:
            objs = self.manager.precache_caches[cache_name][value]
        except KeyError:
            raise CacheIsMissingError()

        for obj in objs:
            yield obj

    def _get_iexact_generator(self, cache_name, value):
        value = value and value.lower()

        for obj in self.manager.precache_caches[cache_name].get(value, []):
            yield obj

    def _get_in_generator(self, cache_name, values):
        default = []
        cache = self.manager.precache_caches[cache_name]

        if cache_name == 'pk_exact':
            for val in values:
                try:
                    yield cache[val][0]
                except KeyError:
                    pass
        else:
            for val in values:
                for obj in cache.get(val, default):
                    yield obj


class PrecachingManager(Manager):
    """
    Кэширует объекты в памяти инстанса для метода get
    Джанга копирует менеджеры при старте
    Помодельный прекеш был вынесен на уровень класса
    https://st.yandex-team.ru/RASPFRONT-7057
    """

    use_for_related_fields = True
    _precache_caches = defaultdict(lambda: defaultdict(dict))
    _precached = defaultdict(lambda: False)

    def __init__(self, keys=None, iexact_keys=None, select_related=None, use_get_fallback=None):
        super(PrecachingManager, self).__init__()

        # Если не нашли в кеше, лезем в базу
        self.use_get_fallback = use_get_fallback

        if keys is None:
            keys = ['pk']

        self.precache_exact_keys = ['pk' if x == 'id' else x for x in keys]
        self.precache_iexact_keys = iexact_keys or []

        self._select_related = select_related

    @property
    def precached(self):
        return PrecachingManager._precached[self.model]

    @property
    def precache_caches(self):
        return PrecachingManager._precache_caches[self.model]

    @contextmanager
    def using_precache(self):
        already_precached = self.precached
        if not already_precached:
            self._fill_cache()

        yield

        if not already_precached:
            self._reset_cache()

    def precache(self):
        self._reset_cache()
        self._fill_cache()

    def _reset_cache(self):
        PrecachingManager._precache_caches[self.model] = defaultdict(dict)
        PrecachingManager._precached[self.model] = False

    def _fill_cache(self):
        self._real_fill_cache()
        PrecachingManager._precached[self.model] = True

    def _real_fill_cache(self):
        assert not self.precached

        use_slots = settings.PRECACHE_USE_PROXY_SLOTS

        objects = self.get_queryset().all()

        def iexact_attrgetter(attr):
            def _attrgetter(obj):
                attr_value = getattr(obj, attr)
                return attr_value and attr_value.lower()

            return _attrgetter

        chunksize = 5000

        if isinstance(objects, QuerySet) and objects.count() > chunksize:
            chunks = model_iter(objects, chunksize=chunksize, in_chunks=True)
        else:
            objects = list(objects)
            chunks = [objects[i * chunksize:(i + 1) * chunksize]
                      for i in range((len(objects) // chunksize) + 1)]

        caches = [
            (attrgetter(field_name), '{}_exact'.format(field_name)) for field_name in self.precache_exact_keys
        ]
        caches.extend([
            (iexact_attrgetter(field_name), '{}_iexact'.format(field_name)) for field_name in self.precache_iexact_keys
        ])

        if use_slots:
            proxificator = get_slots_proxyficator(self.model)

        for chunk in chunks:
            if not isinstance(chunk, list):
                chunk = list(chunk)

            if use_slots:
                chunk = [proxificator(obj) for obj in chunk]

            for field_getter, cache_name in caches:
                cache = self.precache_caches[cache_name]

                cache_update = {}
                for obj in chunk:
                    field_value = field_getter(obj)

                    try:
                        key_update = cache_update[field_value]
                    except KeyError:
                        key_update = cache_update[field_value] = set(cache.get(field_value, []))

                    key_update.add(obj)

                for field_value, objs in six.iteritems(cache_update):
                    cache[field_value] = list(objs)

    def in_bulk_cached(self):
        """
        Этот метод создаваля для ускоренного внедрения прекэша в проект.
        Не используйте его в любом новом коде.
        """
        warnings.warn('[2016-03-28] Do Not Use This Method', RaspDeprecationWarning, stacklevel=2)

        return {o.pk: o for o in self.get_queryset().all()}

    def in_bulk_list(self, pks):
        warnings.warn('[2016-03-28] Use objects.filter(pk__in=list(pks))', RaspDeprecationWarning, stacklevel=2)

        return list(self.get_queryset().filter(pk__in=list(pks)))

    def all_cached(self):
        return list(self.get_queryset().all().order_by())

    def get_cache(self, query_type, key_name):
        return self.precache_caches['{}_{}'.format(key_name, query_type)]

    def get_queryset(self):
        class PrecachingQuerySet(BasePrecachingQuerySet):
            manager = self
            use_get_fallback = self.use_get_fallback

        queryset = PrecachingQuerySet(self.model, using=self._db)

        if self._select_related:
            queryset = queryset.select_related(*self._select_related)

        return queryset

    def __deepcopy__(self, memo):
        """
        Никому не нужна полная копия мэнеджера вместе с кэшами
        """
        return copy.copy(self)
