# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils.datastructures import MultiValueDict
from marshmallow import Schema, fields, missing, post_load, pre_load
from marshmallow.utils import get_value

import six

from travel.rasp.library.python.common23.models.currency.currency import Currency
from travel.rasp.library.python.common23.models.currency.price import Price


class PriceSchema(Schema):
    value = fields.Float()
    currency = fields.Method(serialize='get_currency_iso_code', deserialize='get_currency_as_it_is')

    def get_currency_iso_code(self, data):
        """
        Приводит код валюты к стандартному ISO
        """

        if six.PY3:
            value = get_value(data, 'currency')
        else:
            value = get_value('currency', data)

        if value is not missing:
            # Если код уже в ISO, то можно его сразу вернуть, чтобы избежать лишних хождений в базу
            # RASPFRONT-7588
            if value in {currency.iso_code for currency in Currency.objects.all_cached()}:
                return value

            try:
                return Currency.objects.get(code=value).iso_code
            except Currency.DoesNotExist:
                pass

        return value

    def get_currency_as_it_is(self, currency):
        """
        Валюта должна приходимть как код ISO, в таком виде ее и оставляем
        """
        return currency

    @post_load
    def post_load(self, data, **kwargs):
        if not data:
            return None
        return Price(**data)


class MultiValueDictSchemaMixin(object):
    """
    Получает значения для полей типа List из MultiValueDict. К примеру, из request.GET.
    """
    @pre_load
    def prepare_multivaluedict(self, data, **kwargs):
        if not isinstance(data, MultiValueDict):
            return data

        result = data.dict()
        for field_name, field in self.fields.items():
            if six.PY3:
                load_from = field.data_key or field_name
            else:
                load_from = field.load_from or field_name

            if isinstance(field, fields.List) and load_from in data:
                result[load_from] = data.getlist(load_from)
        return result
