# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import pytz
from datetime import datetime
from six.moves import collections_abc

from django.core.exceptions import ObjectDoesNotExist
from marshmallow import ValidationError, base, fields

from travel.rasp.library.python.common23.models.core.geo.point import Point


class DatetimeAwareField(fields.Field):
    default_error_messages = {
        'invalid': 'Not a valid serialized DatetimeAwareField.',
    }

    def _deserialize(self, value, attr, data, **kwargs):
        try:
            timezone = pytz.timezone(value['tzname'])
            naive_datetime = datetime.strptime(value['local'], '%Y-%m-%d %H:%M:%S')
            return timezone.localize(naive_datetime)
        except (ValueError, KeyError, TypeError, pytz.UnknownTimeZoneError):
            self.fail('invalid')


class PointField(fields.Field):
    default_error_messages = {
        'not_found': 'Point {value} was not found',
        'bad_value': 'Wrong value "{value}" for PointField'
    }

    @property
    def use_hidden_manager(self):
        return True

    def _deserialize(self, value, attr, data, **kwargs):
        try:
            return Point.get_by_key(value, use_hidden_manager=self.use_hidden_manager)
        except ValueError:
            self.fail('bad_value', value=value)
        except ObjectDoesNotExist:
            self.fail('not_found', value=value)

    def _serialize(self, value, attr, obj, **kwargs):
        return value.point_key


class PointShowHiddenField(PointField):
    @property
    def use_hidden_manager(self):
        return False


class DictField(fields.Field):
    """
    Поле для словарей где значения тоже являются полями, в отличие от fields.Dict.

    В схеме использовать следующим образом:
    data_field = DictField(fields.Boolean())
    """
    def __init__(self, child, *args, **kwargs):
        super(DictField, self).__init__(*args, **kwargs)

        if not isinstance(child, base.FieldABC):
            raise ValueError('child must be of type marshmallow.base.FieldABC')

        if isinstance(child, fields.Nested):
            raise ValueError('DictField cannot handle schema inside fields.Nested. Use DictNestedField instead')

        self.child = child

    def _serialize(self, value, attr, obj, **kwargs):
        return {
            key: self.child._serialize(val, key, value)
            for key, val in value.items()
        }

    def _deserialize(self, value, attr, ob, **kwargs):
        return {
            key: self.child.deserialize(val, key, value)
            for key, val in value.items()
        }


class DictNestedField(fields.Nested):
    """
    Поле для словарей где значения имеют свою схему.

    В схеме использовать следующим образом:
    data_field = DictNestedField(SomeSchema)
    """
    def _map(self, method, mapping):
        result = {}
        errors = {}

        for key, val in mapping.items():
            try:
                result[key] = method(val, key, mapping)
            except ValidationError as error:
                errors[key] = error.messages

        if errors:
            raise ValidationError(errors, data=result)
        else:
            return result

    def _serialize(self, value, _attr, _obj, **kwargs):
        if value is None:
            return None
        else:
            return self._map(super(DictNestedField, self)._serialize, value)

    def _deserialize(self, value, _attr, _data, **kwargs):
        if not isinstance(value, collections_abc.Mapping):
            raise ValidationError('Invalid type {} of {!r} value for DictNestedField'
                                  .format(value.__class__.__name__, value))
        return self._map(super(DictNestedField, self)._deserialize, value)
