# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import Schema, missing


class CachingSchema(Schema):
    """
    Схема, которая способна кэшировать результаты сериализации объектов.
    В наследнике нужно переопределить функцию get_cache_key.
    Например, хорошим кандидатом на ключ для объекта из базы будет obj.id.

    Схема из marshmallow при использовании инстанса создает каждую вложенную схему один раз.
    При создании другого инстанса, создается другой инстанс вложенной схемы.
    Поэтому данный класс подходит для вложенных схем, в которых сериализуются
    одни и те же данные несколько раз.
    """
    def dump(self, obj, many=None, **kwargs):
        if not hasattr(self, '_dump_cache'):
            self._dump_cache = {}

        key = self.get_cache_key(obj)
        if key not in self._dump_cache:
            self._dump_cache[key] = super(CachingSchema, self).dump(obj, many=many, **kwargs)
        return self._dump_cache[key]

    def get_cache_key(self, obj):
        raise NotImplementedError('You need to specify caching key.')


def get_defaults_from_schema(schema):
    result = {}
    for fname, field in schema().fields.items():
        if field.default != missing:
            result[fname] = field.default
        elif field.allow_none:
            result[fname] = None
    return result
