# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from datetime import datetime, timedelta

from django.db import models
from django.utils import six
from django.utils.six import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.date.date import get_pytz
from travel.rasp.library.python.common23.date.run_mask import RunMask
from travel.rasp.library.python.common23.models.fields import TrimmedCharField, ThreadCalendarField

log = logging.getLogger(__name__)


@python_2_unicode_compatible
class BaseThreadTariff(models.Model):
    """
    Тариф нитка-станция-станция
    Рассчитываем что этот тариф всегда соответствует нитке в нашей базе.
    Причем была идея импортировать статические тарифы по номеру.
    Это лучше сделать в отдельную таблицу, чтобы не перегружать эту модель логикой.

    t_type и другие ненормальные поля, притянуты сюда чтобы бысто извлекать дополнительные данные при поиске тарифов.
    """
    thread_uid = models.CharField(verbose_name=_('Uid нитки'),
                                  help_text=_('Идентификатор для однозначного определения нитки'),
                                  max_length=100, null=False, db_index=True)
    year_days = ThreadCalendarField(verbose_name=_('Шаблон дней хождения нитки для дней когда действует эта цена'
                                                   ' во временной зоне нитки.'),
                                    default=u'')
    station_from = models.ForeignKey('www.Station', verbose_name=_('Первая станция'),
                                     related_name='route_tariffs_from')
    station_to = models.ForeignKey('www.Station', verbose_name=_('Последняя станция'),
                                   related_name='route_tariffs_to')
    tariff = models.FloatField(_('Цена'))
    currency = models.CharField(_('Валюта'), max_length=10, null=True, default=None, blank=True)
    is_min_tariff = models.BooleanField(_('Это минимальная цена поставщика, могут быть и выше'), default=False)
    url = TrimmedCharField(verbose_name=_('Order Url'),
                           max_length=255, null=False, default='', blank=True)
    data = models.TextField(verbose_name=_(u'Дополнительные данные'),
                            null=False, blank=False, default='')
    settlement_from = models.ForeignKey('www.Settlement', verbose_name=_('Город отправлени'),
                                        null=True, related_name='tariffs_from')
    settlement_to = models.ForeignKey('www.Settlement', verbose_name=_('Город прибытия'),
                                      null=True, related_name='tariffs_to')
    year_days_from = ThreadCalendarField(verbose_name=_('Календарь отправления со станции station_from'
                                                        ' во временной зоне этой станции'),
                                         default=RunMask.EMPTY_YEAR_DAYS)
    time_zone_from = models.CharField(verbose_name=_('Временная зона отправления'), max_length=30)
    time_from = models.TimeField(verbose_name=_('Время отправления с учетом временной зоны'))
    duration = models.IntegerField(verbose_name=_('Время в пути (в минутах).'), null=False, blank=False)
    time_zone_to = models.CharField(verbose_name=_('Временная зона прибытия'), max_length=30)
    time_to = models.TimeField(verbose_name=_('Время прибытия с учетом временной зоны'))
    supplier = models.ForeignKey('www.Supplier', null=False, blank=False, verbose_name=_(u'Поставщик тарифа'))
    number = models.CharField(verbose_name=_('Номер маршрута'), max_length=100, null=False, blank=True, default='')
    t_type = models.ForeignKey('www.TransportType', verbose_name=_('Тип транспорта'), null=False, blank=False)

    def get_order_data(self):
        return json.loads(self.data) if self.data else {}

    def get_local_departure(self, local_date):
        """
        Возвращает местное время отправления, соответствующее дате local_date
        """
        departure = datetime.combine(local_date, self.time_from)
        local_tz = self.station_from.pytz
        departure_tz = get_pytz(self.time_zone_from)

        if local_tz == departure_tz:
            departure_naive = departure
            departure_local = local_tz.localize(departure_naive)
        else:
            for offset in (0, -1, 1):
                departure_naive = departure + timedelta(days=offset)
                departure_local = departure_tz.localize(departure_naive).astimezone(local_tz)
                if departure_local.date() == local_date:
                    break
            else:
                # невероятный случай, когда два часовых пояса отличаются больше чем на сутки
                return None

        if not RunMask.runs_at(self.year_days_from, departure_naive):
            return

        return departure_local

    def get_price(self):
        from travel.rasp.library.python.common23.models.currency.price import Price
        return Price(self.tariff, self.currency)

    def __str__(self):
        return '-'.join([self.thread_uid, self.station_from.title,
                         self.station_to.title, six.text_type(self.tariff)])

    class Meta(object):
        abstract = True
        verbose_name = _('тариф нитка-точка-точка')
        verbose_name_plural = _('тарифы нитка-точка-точка')
        app_label = 'www'
        index_together = [
            ['settlement_from', 'settlement_to'],
            ['settlement_from', 'station_to'],
            ['station_from', 'settlement_to'],
            ['station_from', 'station_to'],
        ]
