# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import six

from django.db import models
from django.utils.html import escape
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.models.fields import CodeCharField, TrimmedCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field


@six.python_2_unicode_compatible
class TariffType(models.Model):
    """
    Типы тарифов
    """

    DEFAULT_ID = 1
    EXPRESS_ID = 2

    USUAL_CATEGORY = 'usual'
    SEASON_TICKET_CATEGORY = 'season_ticket'
    SPECIAL_CATEGORY = 'special'

    L_title = L_field(_('название типа тарифа'), add_local_field=True, max_length=100)
    L_description = L_field(_('подробное описание типа тарифа'), field_cls=models.TextField,
                            base_lang='ru', default='', null=False, blank=True)

    link = TrimmedCharField(_('Ссылка'), max_length=255, blank=True)
    category = models.CharField(
        verbose_name=_('категория'),
        choices=(
            (SEASON_TICKET_CATEGORY, _('Абонемент')),
            (USUAL_CATEGORY, _('Обычный')),
            (SPECIAL_CATEGORY, _('Специальный')),
        ),
        max_length=20, null=False, blank=False,
        default=USUAL_CATEGORY
    )

    code = CodeCharField(verbose_name=_('код'), max_length=20,
                         null=False, blank=False, unique=True)
    order = models.IntegerField(verbose_name=_('порядок показа'),
                                default=0)

    is_main = models.BooleanField('Отображать в сегменте', default=False)

    tariff_groups = models.ManyToManyField('www.TariffGroup', verbose_name=_(u'Группы тарифов'))

    objects = PrecachingManager(keys=['pk', 'code'])

    TANKER_L_FIELDS = ['title', 'description']

    def __str__(self):
        return self.L_title()

    @property
    def title_with_link(self):
        if self.link:
            return '<a href="%s">%s</a>' % (escape(self.link), self.L_title())

        return self.L_title()

    class Meta(object):
        ordering = ('order',)
        verbose_name = _('Тип тарифа')
        verbose_name_plural = _('Типы тарифов')
        app_label = 'www'
