# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import six
from builtins import filter
from builtins import next
from functools import wraps

from django.conf import settings
from django.db import models
from django.utils import translation
from django.utils.functional import lazy
from django.utils.translation import ugettext_lazy as _, ugettext

from travel.rasp.library.python.common23.models.fields.fields import TrimmedCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.utils.text import normalize
from travel.rasp.library.python.common23.xgettext.i18n import markdbtrans


class L_stops_mixin(object):
    def L_stops(self):
        if not getattr(self, 'stops_translations', None):
            return ''

        lang = translation.get_language()

        stops = json.loads(self.stops_translations)

        if lang in stops:
            return stops[lang]

        return ''


class FalseUnicode(six.text_type):
    def __bool__(self):
        return False

    __nonzero__ = __bool__


class Translations(object):
    def __init__(self, field, obj):
        self.field = field
        self.obj = obj

    def __call__(self, case='nominative', lang=None, fallback=True, **kwargs):
        lang = lang or translation.get_language()
        linguistics = (
            self.field.linguistics
            if fallback else
            self.field.exact_linguistics
        )
        form = linguistics(self.obj, lang, case, **kwargs)
        marked = markdbtrans(form)

        if not form and marked:
            return FalseUnicode(marked)

        return marked

    def dict(self):
        return dict(
            (field, getattr(self.obj, field))
            for field in self.field.local_fields()
        )

    def by_language(self):
        if self.field.add_local_field:
            attname = self.field.field_name
            yield None, attname, getattr(self.obj, attname)

        for lang in self.field.LANGUAGES:
            attname = '%s_%s' % (self.field.field_name, lang)

            yield lang, attname, getattr(self.obj, attname)

    @property
    def base_value(self):
        parts = [self.field.field_name]

        if self.field.base_lang:
            parts.append(self.field.base_lang)

        return getattr(self.obj, '_'.join(parts))

    def contains(self, value, transformer=normalize):
        """
        Возвращает True если хотябы одно из локализованных значений равно value
        """
        value = transformer(value)

        for attr_name in self.field.local_fields():
            if transformer(getattr(self.obj, attr_name)) == value:
                return True

        return False

    def __repr__(self):
        return '<%s %r>' % (
            self.__class__.__name__,
            self.dict()
        )


class NoneTranslations(object):
    def __call__(self, *args, **kwargs):
        return u''

    def contains(self, *args, **kwargs):
        return False


class ConstantTranslations(object):
    def __init__(self, value):
        self.value = value

    def __call__(self, *args, **kwargs):
        return self.value

    def contains(self, value, transformer=normalize):
        return transformer(value) == transformer(self.value)


def nonzero_return(default=None):
    def decorator(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            generator = func(*args, **kwargs)
            return next(filter(None, generator), default)

        return wrapped

    return decorator


class L_field(object):
    LANGUAGES = settings.MODEL_LANGUAGES

    def __init__(self, verbose_name,
                 critical=False,
                 add_local_field=False,
                 local_field_critical=False,
                 local_field_unique=False,
                 base_lang=None,
                 base_field_critical=False,
                 add_override_field=False,
                 field_cls=None,
                 extra={},
                 fallback_fields=[],
                 **kwargs):
        self.verbose_name = verbose_name
        self.critical = critical
        self.add_local_field = add_local_field
        self.local_field_critical = local_field_critical
        self.local_field_unique = local_field_unique
        self.base_lang = base_lang
        self.base_field_critical = base_field_critical
        self.add_override_field = add_override_field
        self.field_cls = field_cls or self.char_field
        self.extra = extra
        self.fallback_fields = fallback_fields
        self.kwargs = kwargs

    @classmethod
    def admin_fields(cls, model, fields, group_by='field', **kwargs):
        return tuple(cls._admin_fields(model, fields, group_by, **kwargs))

    @classmethod
    def _admin_fields(cls, model, fields, group_by='field', **kwargs):
        if group_by == 'field':
            for field in fields:
                obj = getattr(model, 'L_' + field)

                if kwargs.get('show_local_field') and obj.add_local_field:
                    yield obj.field_name

                for lang in cls.LANGUAGES:
                    yield '%s_%s' % (obj.field_name, lang)

                    if obj.add_override_field:
                        yield '%s_%s_override' % (obj.field_name, lang)

                    extra_fields = obj.extra.get(lang, [])

                    for case, _field in extra_fields:
                        yield '%s_%s_%s' % (obj.field_name, lang, case)

        elif group_by == 'lang':
            for lang in cls.LANGUAGES:
                for field in fields:
                    obj = getattr(model, 'L_' + field)

                    yield '%s_%s' % (obj.field_name, lang)

                    if obj.add_override_field:
                        yield '%s_%s_override' % (obj.field_name, lang)

                    extra_fields = obj.extra.get(lang, [])

                    for case, _field in extra_fields:
                        yield '%s_%s_%s' % (obj.field_name, lang, case)

        else:
            raise ValueError(u'Unknown group type: %s' % group_by)

    @classmethod
    def search_fields(cls, model, fields):
        return list(cls._search_fields(model, fields))

    @classmethod
    def _search_fields(cls, model, fields):
        for field in fields:
            obj = getattr(model, 'L_' + field)

            if obj.add_local_field:
                yield obj.field_name

            for lang in obj.LANGUAGES:
                yield '%s_%s' % (obj.field_name, lang)

                extra_fields = obj.extra.get(lang, [])

                for case, field_object in extra_fields:
                    yield '%s_%s_%s' % (obj.field_name, lang, case)

    def local_fields(self):
        if self.add_local_field:
            yield self.field_name

        for lang in self.LANGUAGES:
            yield '%s_%s' % (self.field_name, lang)

            extra_fields = self.extra.get(lang, [])

            for case, field_object in extra_fields:
                yield '%s_%s_%s' % (self.field_name, lang, case)

    @classmethod
    def char_field(cls, verbose_name, **extra_kwargs):
        kwargs = dict(default=None, max_length=100, null=True, blank=True, db_index=False)

        kwargs.update(extra_kwargs)

        return TrimmedCharField(verbose_name, **kwargs)

    def contribute_to_class(self, cls, name):
        self.name = name
        self.model = cls

        assert name.startswith('L_')

        self.field_name = name[2:]

        # Connect myself
        setattr(cls, name, self)

        if self.add_local_field:
            kwargs = self.kwargs.copy()

            if self.local_field_critical:
                kwargs['null'] = kwargs['blank'] = False

            kwargs['unique'] = self.local_field_unique

            field = self.field_cls(self.verbose_name, **kwargs)

            cls.add_to_class(self.field_name, field)

        for lang in self.LANGUAGES:
            verbose_name = lazy(
                lambda verbose_name, lang: '%s (%s)' % (ugettext(verbose_name), lang),
                six.text_type
            )(self.verbose_name, lang)

            kwargs = self.kwargs.copy()

            if self.critical or self.base_field_critical and self.base_lang == lang:
                kwargs['null'] = kwargs['blank'] = False

            cls.add_to_class('%s_%s' % (self.field_name, lang), self.field_cls(verbose_name, **kwargs))

            if self.add_override_field:
                cls.add_to_class('%s_%s_override' % (self.field_name, lang),
                                 models.BooleanField(_(u'не обновлять из танкера (%s)' % lang),
                                                     default=False))

        for lang, extra_fields in list(self.extra.items()):
            for case, field_object in extra_fields:
                cls.add_to_class("%s_%s_%s" % (self.field_name, lang, case), field_object)

    def exact_linguistics(self, obj, lang, case, **_kwargs):
        field = self.field_name

        if case == 'nominative':
            attname = '_'.join([field, lang])
        else:
            attname = '_'.join([field, lang, case])

        return getattr(obj, attname, None)

    @nonzero_return(default=u'')
    def linguistics(self, obj, lang, case, field_fallback=True, **kwargs):
        L_fields = [self]

        if field_fallback:
            for field_name in self.fallback_fields:
                field = getattr(self.model, field_name)

                assert isinstance(field, L_field), \
                    "fallback_fields should reference L_field fields"

                L_fields.append(field)

        @nonzero_return()
        def field_fallback_linguistics(*args):
            for field in L_fields:
                yield field.exact_linguistics(obj, *args, **kwargs)

        yield field_fallback_linguistics(lang, case)

        for fallback_lang, fallback_case in settings.LANGUAGE_CASE_FALLBACKS.get((lang, case), []):
            yield field_fallback_linguistics(fallback_lang, fallback_case)

        if case != 'nominative':
            yield field_fallback_linguistics(lang, 'nominative')

        for fallback_lang in settings.LANGUAGE_FALLBACKS.get(lang, []):
            yield field_fallback_linguistics(fallback_lang, 'nominative')

        for field in L_fields:
            if field.add_local_field:
                yield getattr(obj, field.field_name)

        if self.base_lang:
            yield self.exact_linguistics(obj, self.base_lang, 'nominative')

    def __get__(self, instance, instance_type=None):
        if instance is None:
            return self

        return Translations(self, instance)

    def __set__(self, instance, value):
        if instance is None:
            raise AttributeError(u"%s must be accessed via instance" % self.name)

        if isinstance(value, Translations):
            for field in self.local_fields():
                field_value = getattr(value.obj, field)

                setattr(instance, field, field_value)

        else:
            raise ValueError("%r assignment is not supported" % type(value))


class AbstractTranslate(models.Model):
    """
    Перевод значения поля, но в отличае от L_field хранит переведенное значение в отдельной таблице.

    Это может быть полезно в нескольких случаях:

    1) Значение поля объекта повторяется для множества объектов.
    2) Значение поля одинаково для нескольких моделей

    Хранение переводов в отдельной таблице позволяет уменьшить кол-во ручной работы по переводу


    Для работы с полем необходимо описать модель для хранения переводов следующим образом:

    class DirectionTranslate(AbstractTranslate):
        '''
        Переводы названий направлений
        '''

        @classmethod
        def get_keys(cls):
            # Доступные ключи для перевода по объектам модели

        class Meta:
            ordering = ('value',)
            verbose_name = _(u'перевод названия направления')
            verbose_name_plural = _(u'переводы названий направлений')
            app_label = 'www'

    В таблице переводов для колонки value должен быть выставлен collation в utf8_bin (ALTER TABLE table_name
    MODIFY value VARCHAR(100) CHARACTER SET utf8 COLLATE utf8_bin)


    Далее необходимо описать поле с переводами в моделе

    L_title = DirectionTranslate.get_L_method(key_field='title')

    где title - название поля содержащее ключи перевода


    Таблицу переводов необходимо заполнять скриптом fill_translates.py

    Для выгрузки и загрузки в танкер нужно пользоваться dbtanker.py
    """

    TANKER_KEY_FIELD = 'value'
    TANKER_L_FIELDS = ['value']

    L_value = L_field(verbose_name=_(u'значение'), add_local_field=True, local_field_unique=True)
    objects = PrecachingManager(keys=['value'], use_get_fallback=False)

    @classmethod
    def get(cls, value, lang=None):
        value = value and value.strip()

        if value:
            try:
                translate = cls.get_translate(value).L_value(lang=lang)

                if translate:
                    return translate

            except cls.DoesNotExist:
                pass

        return markdbtrans(value)

    @classmethod
    def get_translate(cls, value):
        return cls.objects.get(value=value)

    @classmethod
    def get_L_method(cls, key_field):
        @property
        def method(self):
            key = getattr(self, key_field)

            try:
                return cls.get_translate(key).L_value

            except cls.DoesNotExist:
                return ConstantTranslations(key)

        return method

    @classmethod
    def update(cls):
        """
        Производит обновление доступных ключей
        """

        cls.update_values(cls.get_keys())

    @classmethod
    def get_keys(cls):
        """
        Строит список (set) доступных ключей для перевода по объектам модели
        """

        raise NotImplementedError("get_keys method should be overridden")

    @classmethod
    def update_values(cls, values):
        """
        Добавляет новые значения для перевода

        values - set ключей для перевода
        """

        current = set(cls.objects.values_list('value', flat=True))

        new_values = values - current - set([None, ''])

        for v in new_values:
            cls(value=v, value_ru=v).save()

    class Meta(object):
        abstract = True
