# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
    ICAO TRANSLITERATIONS RECOMMENDED FOR USE BY STATES
    https://www.icao.int/publications/Documents/9303_p3_cons_en.pdf
"""

LAT_UPPER_CHARS = {
    '\u00C1': 'A',  # Á A aкут
    '\u00C0': 'A',  # À A гравис
    '\u00C2': 'A',  # Â A циркумфлекс
    '\u00C4': 'A',  # Ä A диэресис AE или A
    '\u00C3': 'A',  # Ã A тильда
    '\u0102': 'A',  # Ă A бреве
    '\u00C5': 'A',  # Å A кольцо AA или A
    '\u0100': 'A',  # Ā A макрон
    '\u0104': 'A',  # Ą A огонек
    '\u0106': 'C',  # Ć C акут
    '\u0108': 'C',  # Ĉ C циркумфлекс
    '\u010C': 'C',  # Č C карон
    '\u010A': 'C',  # Ċ C точка, акцент
    '\u00C7': 'C',  # Ç C седиль
    '\u0110': 'D',  # Ð Эт
    '\u010E': 'D',  # Ď D карон
    '\u00C9': 'E',  # É E акут
    '\u00C8': 'E',  # È E гравис
    '\u00CA': 'E',  # Ê E циркумфлекс
    '\u00CB': 'E',  # Ë E диэресис
    '\u011A': 'E',  # Ě E карон
    '\u0116': 'E',  # Ė E точка, акцент
    '\u0112': 'E',  # Ē E макрон
    '\u0118': 'E',  # Ę E огонек
    '\u0114': 'E',  # Ĕ E бреве
    '\u011C': 'G',  # Ĝ G циркумфлекс
    '\u011E': 'G',  # Ğ G бреве
    '\u0120': 'G',  # Ġ G точка, акцент
    '\u0122': 'G',  # Ģ G седиль
    '\u0126': 'H',  # Ħ H штрих
    '\u0124': 'H ',  # Ĥ H циркумфлекс
    '\u00CD': 'I',  # Í I акут
    '\u00CC': 'I',  # Ì I гравис
    '\u00CE': 'I',  # Î I циркумфлекс
    '\u00CF': 'I',  # Ï I диэресис
    '\u0128': 'I',  # Ĩ I тильда
    '\u0130': 'I',  # İ I точка, акцент
    '\u012A': 'I',  # Ī I макрон
    '\u012E': 'I',  # Į I огонек
    '\u012C': 'I',  # Ĭ I бреве
    '\u0134': 'J',  # Ĵ J циркумфлекс
    '\u0136': 'K',  # Ķ K седиль
    '\u0141': 'L',  # Ł L слэш
    '\u0139': 'L',  # Ĺ L акут
    '\u013D': 'L',  # Ľ L карон
    '\u013B': 'L',  # Ļ L седиль
    '\u013F': 'L',  # Ŀ L точка
    '\u0143': 'N',  # Ń N акут
    '\u00D1': 'N',  # Ñ N тильда N или NXX
    '\u0147': 'N',  # Ň N карон
    '\u0145': 'N',  # Ņ N седиль
    '\u014B': 'N',  # η Энг
    '\u00D8': 'OE',  # Ø O слэш
    '\u00D3': 'O',  # Ó O акут
    '\u00D2': 'O',  # Ò O гравис
    '\u00D4': 'O',  # Ô O циркумфлекс
    '\u00D6': 'O',  # Ö O диэресис OE или O
    '\u00D5': 'O',  # Õ O тильда
    '\u0150': 'O',  # Ő O двойной акут
    '\u014C': 'O',  # Ō O макрон
    '\u014E': 'O',  # Ŏ O бреве
    '\u0154': 'R',  # Ŕ R акут
    '\u0158': 'R',  # Ř R карон
    '\u0156': 'R',  # Ŗ R седиль
    '\u015A': 'S',  # Ś S акут
    '\u015C': 'S',  # Ŝ S циркумфлекс
    '\u0160': 'S',  # Š S карон
    '\u015E': 'S ',  # Ş S седиль
    '\u0166': 'T',  # Ŧ T штрих
    '\u0164': 'T',  # Ť T карон
    '\u0162': 'T',  # Ţ T седиль
    '\u00DA': 'U',  # Ú U акут
    '\u00D9': 'U',  # Ù U гравис
    '\u00DB': 'U',  # Û U циркумфлекс
    '\u00DC': 'U',  # Ü U диэресис UE или UXX или U
    '\u0168': 'U',  # Ũ U тильда
    '\u016C': 'U',  # Ŭ U бреве
    '\u0170': 'U',  # Ű U двойной акут
    '\u016E': 'U',  # Ů U кольцо
    '\u016A': 'U',  # Ū U макрон
    '\u0172': 'U',  # Ų U огонек
    '\u0174': 'W',  # Ŵ W циркумфлекс
    '\u00DD': 'Y',  # Ý Y акут
    '\u0176': 'Y',  # Ŷ Y циркумфлекс
    '\u0178': 'Y',  # Ÿ Y диэресис
    '\u0179': 'Z',  # Ź Z акут
    '\u017D': 'Z',  # Ž Z карон
    '\u017B': 'Z',  # Ż Z точка
    '\u00FE': 'TH',  # Þ Торн (Исландия)
    '\u00C6': 'AE',  # Æ лигатура AE
    '\u0132': 'IJ',  # Ĳ лигатура IJ
    '\u0152': 'OE',  # Œ лигатура OE
    '\u00DF': 'SS',  # ß двойная s (Германия)

    'Ệ': 'E',
    'Ồ': 'O',
    'Ș': 'S',
    'ı': 'i',
    'Ə': 'E',
    'Ạ': 'A',
    'Ð': 'TH',
    'Ỹ': 'Y',
    'Ư': 'U',
    'Ả': 'A',
    'Ắ': 'A',
    'Ủ': 'U',
    'Ế': 'E',
    'Ộ': 'O',
}

CYR_UPPER_CHARS = {
    '\u0410': 'A',  # А
    '\u0411': 'B',  # Б
    '\u0412': 'V',  # В
    '\u0413': 'G',  # Г
    '\u0414': 'D',  # Д
    '\u0415': 'E',  # Е
    '\u0401': 'E',  # Ё
    '\u0416': 'ZH',  # Ж
    '\u0417': 'Z',  # З
    '\u0418': 'I',  # И
    '\u0406': 'I',  # І
    '\u0419': 'I',  # Й
    '\u041A': 'K',  # К
    '\u041B': 'L',  # Л
    '\u041C': 'M',  # М
    '\u041D': 'N',  # Н
    '\u041E': 'O',  # О
    '\u041F': 'P',  # П
    '\u0420': 'R',  # Р
    '\u0421': 'S',  # С
    '\u0422': 'T',  # Т
    '\u0423': 'U',  # У
    '\u0424': 'F',  # Ф
    '\u0425': 'KH',  # Х
    '\u0426': 'TS',  # Ц
    '\u0427': 'CH',  # Ч
    '\u0428': 'SH',  # Ш
    '\u0429': 'SHCH',  # Щ
    '\u042B': 'Y',  # Ы
    '\u042A': 'IE',  # Ъ
    '\u042D': 'E',  # Э
    '\u042E': 'IU',  # Ю
    '\u042F': 'IA',  # Я
    '\u0476': 'Y',  # V
    '\u0490': 'G',  # Ґ
    '\u045E': 'U',  # ў
    '\u046A': 'U',  # Ѫ
    '\u0192': 'G',  # ƒ
    '\u0452': 'D',  # Ћ
    '\u0405': 'DZ',  # Ѕ
    '\u0408': 'J',  # Ј
    '\u040C': 'K',  # Ќ
    '\u0459': 'LJ',  # Љ
    '\u045A': 'NJ',  # Њ
    '\u04BB': 'C',  # Һ
    '\u040F': 'DZ',  # Џ
    '\u0454': 'IE',  # Є
    '\u0407': 'I',  # Ї
    'Ь': ''
}

ICAO_CHARS = LAT_UPPER_CHARS.copy()
ICAO_CHARS.update(CYR_UPPER_CHARS)
ICAO_CHARS.update({k.lower(): v.lower() for k, v in ICAO_CHARS.items()})


def transliterate(text):
    if not text:
        return text
    result = ''.join([ICAO_CHARS.get(ch, ch) for ch in text])
    return result
