# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.core import validators
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.db.migration_version import check_migration_version
from travel.rasp.library.python.common23.models.fields import TrimmedCharField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field
from travel.rasp.library.python.common23.xgettext.i18n import gettext, tgettext


class DeLuxeTrain(models.Model):
    numbers = TrimmedCharField(verbose_name=_(u'номера'), max_length=1024,
                               help_text=_(u'Список одной строкой с «/» в качестве разделителя.'),
                               validators=[validators.RegexValidator(
                                   u'^[^\r\n]*$', _(u'Удалите переводы строк из списка')
                               )])

    L_title = L_field(_(u'название'),
                      add_local_field=True, base_lang='ru',
                      field_cls=TrimmedCharField, max_length=100, null=False, blank=True,
                      db_index=True)
    deluxe = models.BooleanField(_(u"фирменный"), default=False)
    high_speed = models.BooleanField(_(u"скоростной"), default=False)
    page_path = TrimmedCharField(verbose_name=_(u'страница'), blank=True, help_text=_(u'путь страницы на Расписаниях'),
                                 max_length=1024)

    t_subtype = models.ForeignKey('www.TransportSubtype', verbose_name=_(u'Подтип транспорта'),
                                  blank=True, null=True, default=None)

    if check_migration_version(11):
        wagon_types = TrimmedCharField(
            verbose_name=_(u'типы вагонов'),
            max_length=300, null=True, blank=True,
            help_text=_(u'Список одной строкой с «/» в качестве разделителя.'),
            validators=[validators.RegexValidator(u'^[^\r\n]*$', _(u'Удалите переводы строк из списка'))]
        )
        im_title = models.CharField(
            verbose_name=_(u'название в системе ИМ'),
            max_length=100, null=True, default='', blank=True
        )
        need_check = models.BooleanField(
            verbose_name=_(u'проводить прямую проверку'),
            help_text=_(u'Проводить прямую проверку номеров по типам вагонов и названию в системе ИМ'),
            default=False
        )
        need_reverse_check = models.BooleanField(
            verbose_name=_(u'проводить обратную проверку'),
            help_text=_(u'Проводить обратную проверку номеров по типам вагонов и названию в системе ИМ'),
            default=False
        )
        min_im_records_count = models.IntegerField(
            verbose_name=_(u'порог отсечения для проверок'),
            help_text=_(u'Минимальное, рассматриваемое при проверках число записей в дневном логе ИМ'),
            null=False, default=0
        )

    objects = PrecachingManager(keys=['id'])

    TANKER_L_FIELDS = ('title',)

    _number2deluxe_train = {}

    @classmethod
    def get_by_number(cls, number):
        """
        Возвращает фирменный поезд по его номеру
        """

        def init():
            for train in DeLuxeTrain.objects.all():
                numbers = train.numbers.split('/')

                for n in numbers:
                    cls._number2deluxe_train[n] = train

        if not cls._number2deluxe_train:
            init()

        return cls._number2deluxe_train.get(number)

    def L_title_short(self, lang=None):
        title = self.L_title(lang=lang)

        if title and self.deluxe:
            # Фраза "фирменный поезд «Невский Экспресс»". Сокращение без слова 'поезд'
            return tgettext(u'фирменный «<special-title/>»', special_title=title)

        elif self.deluxe:
            # Фраза "фирменный поезд". Сокращение без слова 'поезд'
            return gettext(u'фирменный')

        elif title:
            return u'«%s»' % title

        return u''

    class Meta(object):
        verbose_name = _(u'фирменный или именной поезд')
        verbose_name_plural = _(u'фирменные и именные поезда')
        app_label = 'www'
        db_table = 'www_deluxetrain'
