# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.core.validators import RegexValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.db.caching import cache_until_switch
from travel.rasp.library.python.common23.models.fields import TrimmedTextField
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.models.texts.i18n import L_field
from travel.rasp.library.python.common23.models.transport.transport_type import TransportType
from travel.rasp.library.python.common23.utils.files.validators import FileSizeValidator, SvgValidator


def transportsubtypeiconfield_generate_filename(t_subtype, filename):
    return 'data/transportsubtype/icon/{}.svg'.format(t_subtype.code)


class TransportSubtypeIconField(models.FileField):
    def __init__(self, *args, **kwargs):
        kwargs.setdefault('upload_to', transportsubtypeiconfield_generate_filename)
        super(TransportSubtypeIconField, self).__init__(*args, **kwargs)


class TransportSubtype(models.Model):
    """Подтип транспорта"""
    MAX_ICON_SIZE_IN_BYTES = 1024 * 1024
    TANKER_L_FIELDS = ['title', 'title_plural', 'station_name', 'station_name_plural', 'title_partner_page']

    file_size_validator = FileSizeValidator(size_limit_in_bytes=MAX_ICON_SIZE_IN_BYTES)
    svg_validator = SvgValidator()
    code_validator = RegexValidator(
        r'^[0-9a-zA-Z-]+$',
        _(u'В коде подтипа транспорта могут учавствовать только английские буквы цифры и знак минус')
    )

    code = models.CharField(verbose_name=_(u'код'), max_length=50, unique=True, validators=[code_validator])
    has_train_tariffs = models.BooleanField(verbose_name=_(u'Продажа билетов через Экспресс-3'), default=False)
    L_title = L_field(_(u'наименование'), base_field_critical=True, base_lang='ru')
    L_title_short = L_field(_(u'короткое наименование'),
                            base_lang='ru',
                            help_text=_(u"без html-тегов и дополнительной информации"))
    L_title_plural = L_field(_(u'наименование множественное'), base_lang='ru')
    L_station_name = L_field(_(u'название станции'), base_lang='ru',
                             help_text=_(u"например, для поездов - 'вокзал'."))
    L_station_name_plural = L_field(_(u'название станций'),
                                    base_lang='ru',
                                    help_text=_(u"например, для поездов - 'вокзалы'."))
    L_title_partner_page = L_field(_(u'название для станицы партнеров'), base_lang='ru')
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'),
                               null=False, blank=False, related_name='subtype')
    color = models.ForeignKey('www.TransportSubtypeColor', verbose_name=_(u'Цвет'), null=True, blank=True, default=None)
    icon = TransportSubtypeIconField(
        _(u'Иконка (svg)'), validators=[file_size_validator, svg_validator],
        null=True, blank=True, default=None
    )
    comment = TrimmedTextField(_(u'Комментарий'), null=False, blank=True, default='')

    use_in_suburban_search = models.BooleanField(
        verbose_name=_("Показывать нитки поездов с таким подтипом в выдаче электричками"), default=False,
        blank=True, db_index=True)
    L_title_suburban = L_field(_('Наименование на выдаче электричек'), default='', blank=True, null=True, base_lang='ru')

    objects = PrecachingManager(keys=['pk', 'code'])

    TRAIN_ID = 1
    PLANE_ID = 2
    BUS_ID = 3
    RIVER_ID = 4
    SEA_ID = 5
    SUBURBAN_ID = 6
    HELICOPTER_ID = 7
    LAST_SUBURBAN_ID = 22
    LASTDAL_ID = 43

    @classmethod
    def get_default_subtype_id(cls, t_type_id):
        return {
            TransportType.TRAIN_ID: cls.TRAIN_ID,
            TransportType.PLANE_ID: cls.PLANE_ID,
            TransportType.BUS_ID: cls.BUS_ID,
            TransportType.SUBURBAN_ID: cls.SUBURBAN_ID
        }.get(t_type_id)

    def __unicode__(self):
        return self.L_title()

    def L_title_without_html_symbols(self, lang=None):
        title = self.L_title(lang=lang) if lang else self.L_title()
        return title.replace('<br/>', ' ').replace('&nbsp;', ' ')

    @classmethod
    @cache_until_switch
    def get_train_search_codes(cls):
        return [obj.code for obj in cls.objects.filter(use_in_suburban_search=True)]

    class Meta(object):
        verbose_name = _(u'подтип транспорта')
        verbose_name_plural = _(u'подтипы транспорта')
        app_label = 'www'
        db_table = 'www_transportsubtype'
