# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings
from django.core.validators import RegexValidator
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.db.caching import cache_until_switch
from travel.rasp.library.python.common23.models.texts.i18n import L_field
from travel.rasp.library.python.common23.models.precache.manager import PrecachingManager
from travel.rasp.library.python.common23.xgettext.i18n import mark_gettext


TRANSPORT_TYPES = {
    'express': mark_gettext(u'экспресс'),
    'aeroexpress': mark_gettext(u'аэроэкспресс'),
    'plane': mark_gettext(u'самолёт'),
    'train': mark_gettext(u'поезд'),
    'bus': mark_gettext(u'автобус'),
    'suburban': mark_gettext(u'электричка'),
    'sea': mark_gettext(u'морской транспорт'),
    'river': mark_gettext(u'речной транспорт'),
    'water': mark_gettext(u'водный транспорт')
}


class TransportType(models.Model):
    """ Тип транспорта: train, plane, .. """

    TANKER_L_FIELDS = ['title', 'title_plural', 'station_name', 'station_name_plural', 'title_partner_page']

    code_validator = RegexValidator(
        r'^[0-9a-zA-Z-]+$',
        _(u'В коде типа транспорта могут учавствовать только английские буквы цифры и знак минус')
    )

    code = models.CharField(verbose_name=_(u'код'), max_length=50, unique=True, validators=[code_validator])
    L_title = L_field(_(u'наименование'), base_field_critical=True, base_lang='ru')
    L_title_plural = L_field(_(u'наименование множественное'), base_lang='ru')
    L_station_name = L_field(_(u'название станции'), base_lang='ru',
                             help_text=_(u"например, для поездов - 'вокзал'."))
    L_station_name_plural = L_field(_(u'название станций'), base_lang='ru',
                                    help_text=_(u"например, для поездов - 'вокзалы'."))
    L_title_partner_page = L_field(_(u'название для страницы партнеров'), base_lang='ru')

    objects = PrecachingManager(keys=['pk', 'code'])

    # Константы, чтобы не хардкодить в коде числа
    TRAIN_ID = 1
    PLANE_ID = 2
    BUS_ID = 3
    RIVER_ID = 4
    SEA_ID = 5
    SUBURBAN_ID = 6
    HELICOPTER_ID = 7
    URBAN_ID = 8
    PSEUDO_GORTRANS = 9
    BLABLACAR = 10
    WATER_ID = 11

    RAILWAY_TTYPE_IDS = {TRAIN_ID, SUBURBAN_ID}
    RAILWAY_TTYPE_CODES = {'train', 'suburban'}
    WATER_TTYPE_IDS = [RIVER_ID, SEA_ID, WATER_ID]
    WATER_TTYPE_CODES = ['river', 'sea', 'water']

    def __repr__(self):
        return '{}: {}'.format(self.__class__.__name__, self.code)

    def __unicode__(self):
        return self.L_title()

    @property
    def name(self):
        return self.L_title()

    @classmethod
    @cache_until_switch
    def get_train_type(cls):
        return cls.objects.get(id=cls.TRAIN_ID)

    @classmethod
    @cache_until_switch
    def get_suburban_type(cls):
        return cls.objects.get(id=cls.SUBURBAN_ID)

    @classmethod
    @cache_until_switch
    def get_bus_type(cls):
        return cls.objects.get(id=cls.BUS_ID)

    @classmethod
    @cache_until_switch
    def get_plane_type(cls):
        return cls.objects.get(id=cls.PLANE_ID)

    @staticmethod
    def list_national_codes(national_version):
        """ Коды транспорта для национальной версии """

        return settings.NATIONAL_VERSION_TRANSPORTS.get(
            national_version,
            settings.NATIONAL_VERSION_TRANSPORTS['*']
        )

    type = 'transporttype'

    class Meta(object):
        verbose_name = _(u'тип транспорта')
        verbose_name_plural = _(u'типы транспорта')
        app_label = 'www'
        db_table = 'www_transporttype'
