# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Эти хелперы используются в абстрактных моделях и поэтому не должны зависеть от обычных моделей
"""

from builtins import zip
from builtins import map
import logging
import operator
import random
from itertools import islice

from django.db import models
from django.utils.datastructures import MultiValueDict
from django.utils.translation import ugettext_lazy as _

from travel.rasp.library.python.common23.db.mysql_try_hard import mysql_try_hard


log = logging.getLogger(__name__)

TABLO_MANUAL_STATUS_CHOISES = (
    ('use_real_time', _(u"Вичислить по реальному времени")),
    ('cancelled', _(u"Отменен")),
    ('hidden', _(u"Скрыт"))
)
EVENT_CHOISES = (
    ('arrival', _(u"Прибытие")),
    ('departure', _(u"Отправление"))
)


def take(n, iterable):
    "Return first n items of the iterable as a list"
    return list(islice(iterable, n))


def model_iter(qs, chunksize=100, in_chunks=False, shuffle=False, plain_qs=None):
    pks = list(qs.values_list('pk', flat=True))

    if shuffle:
        random.shuffle(pks)

    plain_qs = plain_qs if plain_qs is not None else qs.all().order_by()

    if chunksize == 1:
        for pk in pks:
            yield plain_qs.get(pk=pk)

        return

    pks_iter = iter(pks)

    while 1:
        pks = take(chunksize, pks_iter)

        if not pks:
            break

        objects = mysql_try_hard(plain_qs.in_bulk)(pks)

        if in_chunks:
            yield [objects[pk] for pk in pks if pk in objects]
        else:
            for pk in pks:
                try:
                    yield objects[pk]
                except KeyError:
                    raise


def _iter_fetch_related(objects, field, model=None, skip_errors=False):
    if model is None:
        model = getattr(objects, 'model', None)

        if model is None:
            raise ValueError('Please pass queryset or specify model directly')

    model_field = getattr(model, field).field
    related_model = model_field.rel.to

    getter = operator.attrgetter(model_field.attname)

    ids = list(map(getter, objects))
    uniq_ids = list(set(ids) - {None})

    related_objects = related_model.objects.order_by().in_bulk(uniq_ids)

    for obj, related_id in zip(objects, ids):
        try:
            setattr(
                obj, field,
                None if related_id is None else related_objects[related_id]
            )
        except Exception:
            log.exception(
                u'Ошибка установки атрибута %s %s %s %s',
                obj.__class__, obj.pk, field,
                related_objects.get(related_id)
            )

            if not skip_errors:
                raise
        else:
            yield obj


def fetch_related(objects, *fields, **kwargs):
    """ Устанавливает related objects через отдельные запросы """

    for field in fields:
        for _obj in _iter_fetch_related(objects, field,
                                        skip_errors=False, **kwargs):
            pass


def fetch_related_safe(objects, *fields, **kwargs):
    """
        Устанавливает related objects через отдельные запросы, пропуская
        объекты, для которых это не получилось сделать
    """

    for field in fields:
        objects = list(_iter_fetch_related(objects, field,
                                           skip_errors=True, **kwargs))

    return objects


def fetch_children(objects, qs, field):
    pk_getter = operator.attrgetter(getattr(qs.model, field).field.attname)

    d = MultiValueDict()

    for child in qs.filter(**{'%s__in' % field: objects}):
        d.appendlist(pk_getter(child), child)

    return d


class HiddenManagerWrapper(models.Manager):
    """
    Взывает к менеджерам, не возвращая скрытые объекты
    """

    def __init__(self, manager_name):
        super(HiddenManagerWrapper, self).__init__()
        self.original_manager_name = manager_name

    @property
    def original_manager(self):
        return getattr(self.model, self.original_manager_name)

    def get(self, *args, **kwargs):
        if getattr(self.original_manager, 'precached', False):
            obj = self.original_manager.get(*args, **kwargs)
            if obj.hidden:
                raise self.model.DoesNotExist
            else:
                return obj

        return self.get_queryset().get(*args, **kwargs)

    def get_list(self, *args, **kwargs):
        return [o for o in self.original_manager.filter(*args, **kwargs) if not o.hidden]

    def get_queryset(self):
        return self.original_manager.get_queryset().filter(hidden=False)
