# coding: utf8
from __future__ import unicode_literals

import os
from collections import OrderedDict

import six
import yenv

from travel.rasp.library.python.common23.settings.configuration.base import BaseConfigurator  # noqa
from travel.rasp.library.python.common23.settings.configuration.cache import (  # noqa
    CacheConfigurator, ReplicaStateCacheConfigurator, BaseRedisCacheConfigurator
)
from travel.rasp.library.python.common23.settings.configuration.misc import MiscConfigurator
from travel.rasp.library.python.common23.settings.configuration.mongo import (  # noqa
    AllMongoDatabasesConfigurator, DefaultMongoConfigurator, DynamicSettingsMongoConfigurator,
    SuburbanEventsMongoConfigurator, TrainPurchaseMongoConfigurator
)
from travel.rasp.library.python.common23.settings.configuration.mysql import MysqlConfigurator
from travel.rasp.library.python.common23.settings.configuration.partners import PartnersConfigurator
from travel.rasp.library.python.common23.settings.configuration.rasp_backend import RaspBackendConfigurator
from travel.rasp.library.python.common23.utils.dcutils import resource_explorer


COMMON_CONFIGURATORS = OrderedDict([
    ('mysql', MysqlConfigurator),
    ('mongo_all', AllMongoDatabasesConfigurator),
    ('replica_state_cache', ReplicaStateCacheConfigurator),
    ('cache', CacheConfigurator),
    ('partners', PartnersConfigurator),
    ('misc', MiscConfigurator),
    ('rasp_backend', RaspBackendConfigurator)
])


class Configuration(object):
    PRODUCTION = 'production'
    TESTING = 'testing'
    STRESS = 'stress'
    DEVELOPMENT = 'development'
    CONTENT = 'content'

    configurator_classes = list(COMMON_CONFIGURATORS.keys())

    def __init__(self, configurator_classes=None):
        self.configurator_classes = configurator_classes or self.configurator_classes

    def apply(self, settings):
        yandex_environment_type = settings.get('YANDEX_ENVIRONMENT_TYPE', yenv.type)

        yandex_environment_name = settings.get('YANDEX_ENVIRONMENT_NAME', yenv.name)

        settings['YANDEX_ENVIRONMENT_TYPE'] = yandex_environment_type
        settings['YANDEX_ENVIRONMENT_NAME'] = yandex_environment_name

        resource_explorer.init()
        if 'YANDEX_DATA_CENTER' not in settings:
            if os.environ.get('RASP_IS_BUILDING') or os.environ.get('YA_TEST_RUNNER'):
                settings['YANDEX_DATA_CENTER'] = 'xxx'
            else:
                settings['YANDEX_DATA_CENTER'] = resource_explorer.get_current_dc()

        configurators = []
        for configurator_class in self.configurator_classes:
            if isinstance(configurator_class, six.string_types):
                configurators.append(COMMON_CONFIGURATORS[configurator_class](self))
            else:
                configurators.append(configurator_class(self))

        if yandex_environment_type == 'stress':
            settings['APPLIED_CONFIG'] = Configuration.STRESS

        elif yandex_environment_type == 'testing':
            settings['APPLIED_CONFIG'] = Configuration.TESTING

        elif yandex_environment_type == 'development':
            settings['APPLIED_CONFIG'] = Configuration.DEVELOPMENT

        elif yandex_environment_type == 'production':
            settings['APPLIED_CONFIG'] = Configuration.PRODUCTION

        else:
            raise Exception("Environment type {!r} not supported".format(yandex_environment_type))

        for configurator in configurators:
            configurator.apply(settings['APPLIED_CONFIG'], settings)
