# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import os

from rasp_vault.api import get_secret

from travel.rasp.library.python.common23.settings.configuration.base import BaseConfigurator
from travel.rasp.library.python.common23.settings.utils import get_redis_caches_config, bool_converter

log = logging.getLogger(__name__)


_FALLBACK_HOSTS = {
    'production': {
        'sas': [
            'main-1.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-2.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-3.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-4.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-5.main.production.memcached.rasp.stable.qloud-d.yandex.net',
        ],
        'iva': [
            'main-6.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-10.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-11.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-12.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-13.main.production.memcached.rasp.stable.qloud-d.yandex.net',
        ],
        'myt': [
            'main-7.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-8.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-9.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-15.main.production.memcached.rasp.stable.qloud-d.yandex.net',
            'main-16.main.production.memcached.rasp.stable.qloud-d.yandex.net',
        ]
    },
    'testing': {
        'sas': ['main-1.main.testing.memcached.rasp.stable.qloud-d.yandex.net'],
        'iva': ['main-2.main.testing.memcached.rasp.stable.qloud-d.yandex.net'],
        'myt': ['main-3.main.testing.memcached.rasp.stable.qloud-d.yandex.net']
    },
    'stress': {
        'myt': ['main-1.main.stress.memcached.rasp.stable.qloud-d.yandex.net']
    }
}


REDIS_FALLBACK_HOSTS = {
    'production': [
        'sas-wdbc7w1zxsfng7gu.db.yandex.net',
        'sas-p5qmuxgd7r27wkma.db.yandex.net',
        'man-u1t9u3el4b50satq.db.yandex.net',
        'myt-0nkrezrxwadusc0x.db.yandex.net',
        'vla-s8ebmf149hatmmxd.db.yandex.net'
    ],
    'testing': [
        'myt-dipz25ixgbvm9pdw.db.yandex.net',
        'sas-55wl4mixntgu4rqg.db.yandex.net',
        'vla-vryx69z6q518m2v2.db.yandex.net'
    ],
    'stress': []
}


class ReplicaStateCacheConfigurator(BaseConfigurator):
    def apply_base(self, settings):
        settings['CACHES'].update({
            settings['REPLICA_STATE_CACHE']: {
                'BACKEND': 'django.core.cache.backends.locmem.LocMemCache',
                'timeout': 60,
            }
        })


class BaseRedisCacheConfigurator(BaseConfigurator):
    CLUSTER_ID_SECRET = None
    CLUSTER_PASSWORD_SECRET = None
    CLUSTER_NAME_SECRET = None
    CACHE_ALIAS = None
    FALLBACK_HOSTS = None

    def configure_redis(self, settings, secret):
        redis_cluster_id = secret[self.CLUSTER_ID_SECRET]
        redis_cluster_password = secret[self.CLUSTER_PASSWORD_SECRET]
        redis_cluster_name = secret[self.CLUSTER_NAME_SECRET]

        settings['CACHES'].update(get_redis_caches_config(
            cluster_id=redis_cluster_id,
            cluster_password=redis_cluster_password,
            cluster_name=redis_cluster_name,
            fallback_hosts=self.FALLBACK_HOSTS.get(settings['APPLIED_CONFIG']),
            alias=self.CACHE_ALIAS
        ))


# Когда избавимся от RASP_ENABLE_REDIS_CACHE нужно будет унаследовать caches['default'] от BaseRedisCacheConfigurator
class CacheConfigurator(BaseConfigurator):
    def configure_redis(self, settings, secret):
        redis_cluster_id = secret['REDIS_DEFAULT_CLUSTER_ID']
        redis_cluster_password = secret['REDIS_DEFAULT_PASSWORD']
        redis_cluster_name = secret['REDIS_DEFAULT_NAME']

        settings['CACHES'].update(get_redis_caches_config(
            cluster_id=redis_cluster_id,
            cluster_password=redis_cluster_password,
            cluster_name=redis_cluster_name,
            fallback_hosts=REDIS_FALLBACK_HOSTS.get(settings['APPLIED_CONFIG'])
        ))

    def use_redis(self, settings):
        settings['ENABLE_REDIS_CACHE'] = settings.get('ENABLE_REDIS_CACHE') or bool_converter(os.getenv('RASP_ENABLE_REDIS_CACHE'))
        return settings['ENABLE_REDIS_CACHE']

    def apply_stress(self, settings):
        if not settings.get('DISABLE_GROUP_CACHES'):
            self.configure_redis(settings, get_secret('rasp-common-stress'))

    def apply_testing(self, settings):
        if not settings.get('DISABLE_GROUP_CACHES'):
            self.configure_redis(settings, get_secret('rasp-common-testing'))

    def apply_production(self, settings):
        if not settings.get('DISABLE_GROUP_CACHES'):
            self.configure_redis(settings, get_secret('rasp-common-production'))
