# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from builtins import map
import datetime
import logging
import os
import socket
import warnings
from collections import defaultdict
from copy import deepcopy
from itertools import chain

from django.core.exceptions import ImproperlyConfigured
from rasp_vault.api import get_secret

from travel.rasp.library.python.api_clients.mdb import MdbApiWrapper, REDIS_API_BASE_URL
from travel.rasp.library.python.common23.utils.dcutils import resource_explorer, ResourceExplorer
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning

nothing = object()


log = logging.getLogger(__name__)


def get_setting(name, env=None, default=nothing):
    warnings.warn('[2017-10-20] Use define_setting', RaspDeprecationWarning, stacklevel=2)
    from django.conf import settings

    applied_config = getattr(settings, 'APPLIED_CONFIG', None)
    if env and applied_config in env:
        default = env[applied_config]

    if default is nothing:
        return getattr(settings, name)
    else:
        return getattr(settings, name, default)


def define_setting(name, env=None, default=nothing, converter=lambda x: x):
    """
    Объявляем настройку name.
    Настройка будет доступна на django.conf.settings.
    Значения по умолчанию можно переопределить, если положить настройку в setting до define,
    например в local_settings.
    """
    from django.conf import settings

    def _set_setting(name, value):
        # Если настройки переопределены, то добавляем нашу настройку в корневой setting
        # Такое может случиться если define прошел в контексте override_settings
        # Так же могут быть вложенные override_settings
        target_settings = settings
        while hasattr(target_settings, 'default_settings'):
            target_settings = target_settings.default_settings
        setattr(target_settings, name, value)

    if os.environ.get('RASP_' + name) is not None:
        value = converter(os.environ.get('RASP_' + name))
        _set_setting(name, value)
        return

    if getattr(settings, name, nothing) is not nothing:
        return

    applied_config = getattr(settings, 'APPLIED_CONFIG', None)
    if env and applied_config in env:
        default = env[applied_config]

    assert default is not nothing

    if callable(default):  # для получения значения только в конкретном окружении (например, получение секрета в проде)
        default = default()

    _set_setting(name, default)


def get_cache_config(hosts):
    return {
        'BACKEND': 'travel.rasp.library.python.common23.db.memcache_backend.MemcachedCache',
        'LOCATION': list(map('{}:11211'.format, hosts)),
        'TIMEOUT': 120,
        'LONG_TIMEOUT': 24 * 60 * 60,
        'OPTIONS': {
            'server_max_value_length': 128 * 1024 * 1024,
            'socket_timeout': 2
        }
    }


def get_redis_cache_config(hosts, password, name, current_dc):
    return {
        'BACKEND': 'django_redis.cache.RedisCache',
        'LOCATION': [  # первым в списке должен быть мастер
            'redis://{}:6379'.format(host)
            for host in hosts
        ],
        'OPTIONS': {
            'CLIENT_CLASS': 'travel.rasp.library.python.common23.db.redis.django_redis_sentinel_client.DjangoRedisSentinelClient',
            'SENTINEL_SERVICE_NAME': name,
            'SENTINEL_HOSTS': [(host, 26379) for host in hosts],
            'SENTINEL_SOCKET_TIMEOUT': 0.2,
            'CURRENT_DC': current_dc,
            'PASSWORD': password,
            'SOCKET_TIMEOUT': 2,
        },
        'TIMEOUT': 120,
        'LONG_TIMEOUT': 24 * 60 * 60,
    }


def get_redis_caches_config(cluster_id, cluster_password, cluster_name, fallback_hosts=None, alias='default'):
    mdb_redis_api = MdbApiWrapper(
        api_base_url=REDIS_API_BASE_URL,
        oauth_token=get_secret('rasp-common.MDB_OAUTH_TOKEN')
    )

    current_dc = resource_explorer.get_current_dc()
    cache_hosts = defaultdict(list)

    try:
        cluster_info = mdb_redis_api.get_cluster_info(cluster_id)
        if not cluster_info.instances or cluster_info.master is None:
            raise ImproperlyConfigured(
                'Cluster {} ImproperlyConfigured. Cluster: {}'.format(cluster_id, cluster_info.instances)
            )

        hosts = [cluster_info.master.hostname]

        for host in cluster_info.replicas:
            cache_hosts[host.dc].append(host.hostname)

        if cache_hosts:
            if current_dc in cache_hosts:
                current_dc_hosts = cache_hosts.get(current_dc)
                hosts.extend(current_dc_hosts)
            else:
                hosts.extend(chain(*cache_hosts.values()))
    except Exception:
        if fallback_hosts:
            log.exception('get_redis_caches_config failed. cluster: {}'.format(cluster_id))
            hosts = fallback_hosts
        else:
            raise

    return {
        alias: get_redis_cache_config(hosts, cluster_password, cluster_name, current_dc)
    }


def _first_host(hosts):
    if not hosts:
        return None
    return sorted(hosts)[0]


def merge_db_settings(settings_base, settings_new):
    settings_new = deepcopy(settings_new)
    cluster_conf = settings_new.pop('CLUSTER', {})
    settings_base.update(settings_new)
    settings_base.setdefault('CLUSTER', {}).update(cluster_conf)


def apply_switch_workflow(settings, maint_conf, main0_conf, main1_conf, mdb_api_call_enabled=False):
    """
    Настройки для схемы с переключением баз:
     - сервисы должны ходить на конкретный кластер в зависимости от своей роли (work_db, service_db)
     - есть база maintenance, в которой хранятся соответствия роль -> алиас кластера (напр, work_db -> rasp_3)
        - эти соответствия меняются при переключении баз
     - алиасы work_db, service_db ссылаются на алиасы конкретных кластеров
     - алиас default ссылается на work_db/service_db в зависимости от текущей роли сервиса (обычно это work_db)\

    Про mdb_api_call_enabled=False - по умолчанию мы не хотим ходить постоянно в АПИ MDB, вместо этого храним список хостов в yav
    """

    databases = settings['DATABASES']
    databases['default'] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.alias_proxy',
        'ALIAS_GETTER': 'travel.rasp.library.python.common23.db.switcher.instance_role_alias',  # -> work or service
    }

    databases[settings['WORK_DB']] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.alias_proxy',
        'ALIAS_GETTER': 'travel.rasp.library.python.common23.db.switcher.work_alias',  # -> main0_rasp or main1_rasp
    }

    databases[settings['SERVICE_DB']] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.alias_proxy',
        'ALIAS_GETTER': 'travel.rasp.library.python.common23.db.switcher.service_alias',  # -> main0_rasp or main1_rasp
    }

    # maintenance
    settings['MAINTENANCE_DB_ENABLED'] = True
    databases[settings['MAINTENANCE_DB']] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.cluster_mdb',
        'STORAGE_ENGINE': 'InnoDB',
        'NAME': 'rasp_maintenance',
        'CLUSTER': {
            'CLUSTER_ID': None,
            'MDB_API_CALL_ENABLED': mdb_api_call_enabled,
            'CHECK_MASTER_ON_EACH_CONNECT': False,
        }
    }
    merge_db_settings(databases[settings['MAINTENANCE_DB']], maint_conf)

    # main0
    databases['main0_rasp'] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.cluster_mdb',
        'STORAGE_ENGINE': 'InnoDB',
        'NAME': 'rasp',
        'CLUSTER': {
            'CLUSTER_ID': None,
            'MDB_API_CALL_ENABLED': mdb_api_call_enabled,
            'CHECK_MASTER_ON_EACH_CONNECT': False,
        }
    }
    merge_db_settings(databases['main0_rasp'], main0_conf)

    # main1
    databases['main1_rasp'] = {
        'ENGINE': 'travel.rasp.library.python.common23.db.backends.cluster_mdb',
        'STORAGE_ENGINE': 'InnoDB',
        'NAME': 'rasp',
        'CLUSTER': {
            'CLUSTER_ID': None,
            'MDB_API_CALL_ENABLED': mdb_api_call_enabled,
            'CHECK_MASTER_ON_EACH_CONNECT': False,
        }
    }
    merge_db_settings(databases['main1_rasp'], main1_conf)


def get_fallback_conf_from_str(hosts_str):
    """
    Список хостов через запятую -> конфигурация FALLBACK для db.backends.cluster_mdb
    Используется для хранения хостов в Секретнице
    """
    if not hosts_str:
        return

    hosts = [host.strip() for host in hosts_str.split(',')]
    master, replicas = hosts[0], hosts[1:]

    return {
        'MASTER': master,
        'REPLICAS': replicas,
    }


def use_master_only(settings, aliases=None):
    databases = settings['DATABASES']

    if aliases is None:
        aliases = databases.keys()

    for alias in aliases:
        cluster_conf = databases[alias].get('CLUSTER')
        if cluster_conf:
            cluster_conf['USE_MASTER'] = True
            cluster_conf['USE_REPLICAS'] = False
            cluster_conf['CHECK_MASTER_ON_EACH_CONNECT'] = True


def get_base_domain(environment_type, environment_name):
    BASE_DOMAINS = {
        'wizard01h.tst.rasp.yandex.net': 'lepus',
        'raspadmin-test.rasp.yandex.net': 'admin-test.rasp',
        'avia-teamcity-buildfarm': 'avia-teamcity-buildfarm.haze',
    }

    hostname = socket.gethostname()

    try:
        return BASE_DOMAINS[hostname]
    except KeyError:
        pass

    if (
        environment_type == 'testing' and
        environment_name == 'localhost' and
        not hostname.endswith('yandex.net')
    ):
        return hostname + '.dev.rasp'

    if hostname.endswith('.rasp.yandex.net'):
        return hostname[:-len('.rasp.yandex.net')]

    if hostname.endswith('.yandex.net'):
        return hostname[:-len('.yandex.net')]

    if hostname.endswith('.yandex.ru'):
        return hostname[:-len('.yandex.ru')]

    raise Exception("Could not determine base_domain for host %r" % hostname)


def bool_converter(value):
    if not value:
        return False
    if value.lower() in ('t', '1', 'true', 'y', 'yes'):
        return True
    if value.lower() in ('f', '0', 'false', 'n', 'no'):
        return False
    raise Exception('Bad bool value {!r}'.format(value))


def configure_raven(settings, service_name):
    settings['SENTRY_CLIENT'] = 'travel.rasp.library.python.common23.data_api.error_booster.raven.RavenErrorBoosterClient'
    settings.setdefault('RAVEN_CONFIG', {}).update({
        'service_name': service_name,
        'transport': 'travel.rasp.library.python.common23.data_api.error_booster.raven.RavenErrorBoosterTransport',
        'environment': settings['APPLIED_CONFIG']
    })


def get_app_version():
    if ResourceExplorer.is_run_in_qloud():
        version = os.environ['QLOUD_DOCKER_IMAGE'].rsplit(':', 1)[-1]
        return os.environ['QLOUD_DOCKER_HASH'] if version == 'latest' else version
    elif ResourceExplorer.is_run_in_deploy():
        version = os.environ['DEPLOY_DOCKER_IMAGE'].rsplit(':', 1)[-1]
        return os.environ['DEPLOY_DOCKER_HASH'] if version == 'latest' else version
    else:
        return datetime.datetime.today().strftime('dev.%Y%m%d%H%M%S')
