# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Писать в удаленный кэш нужно всегда в потоке https://st.yandex-team.ru/EXRASP-11023
"""

from functools import wraps

from django.conf import settings
from django.core.cache import cache
from django.utils.translation import get_language


def memoize(func):
    """
    Кеширует результаты выполнения функции, обновляет результаты если сменилась база.
    Все аргументы должны быть hashable.
    """
    memoized = {}

    @wraps(func)
    def new_func(*args, **kwargs):
        key = args + tuple([(k, v) for k, v in kwargs.items()])
        try:
            return memoized[key]
        except KeyError:
            memoized[key] = func(*args, **kwargs)
            return memoized[key]

    return new_func


def cache_method_result(method):
    method_name = method.__name__

    cache_attr = '_cache_' + method_name

    @wraps(method)
    def wrapper(self, *args, **kwargs):
        try:
            cache = getattr(self, cache_attr)
        except AttributeError:
            cache = dict()
            setattr(self, cache_attr, cache)

        key = args + tuple([(k, v) for k, v in kwargs.items()])

        if key in cache:
            return cache[key]
        else:
            return cache.setdefault(key, method(self, *args, **kwargs))

    return wrapper


def cache_method_result_with_exception(method):
    method_name = method.__name__

    cache_attr = '_cache_' + method_name
    exceptions_cache_attr = '_exceptions_' + method_name

    @wraps(method)
    def wrapper(self, *args, **kwargs):
        try:
            cache = getattr(self, cache_attr)
            exception_cache = getattr(self, exceptions_cache_attr)
        except AttributeError:
            cache = dict()
            exception_cache = dict()

            setattr(self, cache_attr, cache)
            setattr(self, exceptions_cache_attr, exception_cache)

        key = args + tuple([(k, v) for k, v in kwargs.items()])
        try:
            return cache[key]
        except KeyError:
            pass

        if key in exception_cache:
            raise exception_cache[key]

        try:
            cache[key] = method(self, *args, **kwargs)
            return cache[key]
        except Exception as e:
            exception_cache[key] = e
            raise

    return wrapper


def cached(key_func, timeout=settings.CACHES['default']['LONG_TIMEOUT'],
           cache_root=settings.CACHEROOT, is_localized=False):
    """
    Кеширующий декоратор.
    is_localized - если значение True, то результат вызова декорируемой функции
                   зависит от текущего языка запроса
    """

    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            key = cache_root + key_func(*args, **kwargs)

            if is_localized:
                key += '/' + get_language()

            if key is None:
                return func(*args, **kwargs)  #: не кешируем запрос

            value = cache.get(key)

            if not value:
                value = func(*args, **kwargs)
                cache.set(key, value, timeout)

            return value

        return wrapper
    return decorator


def get_package_cache_root():
    return '/yandex/rasp/{}/'.format(settings.PKG_VERSION)
