# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os.path
import socket


class ResourceExplorer(object):
    @classmethod
    def is_run_in_qloud(cls):
        return bool(os.environ.get('QLOUD_PROJECT'))

    @classmethod
    def is_run_in_deploy(cls):
        return bool(os.environ.get('TRAVEL_DEPLOY_PROJECT'))

    @classmethod
    def is_run_in_sandbox(cls):
        return bool(os.environ.get('SANDBOX_CONFIG'))

    @classmethod
    def is_run_in_qyp(cls):
        hostname = socket.gethostname()
        return hostname.endswith('.yp-c.yandex.net')

    def get_current_dc(self):
        ya_deploy_dc = os.environ.get('DEPLOY_NODE_DC')  # https://wiki.yandex-team.ru/deploy/dev/envvars/
        if ya_deploy_dc:
            return ya_deploy_dc.lower()

        if os.environ.get('RASP_SANDBOX_DC'):
            return os.environ['RASP_SANDBOX_DC'].lower()

        if self.is_run_in_qloud():
            return os.environ['QLOUD_DATACENTER'].lower()

        if self.is_run_in_qyp():
            hostname = socket.gethostname()
            # format: [name].[dc].yp-c.yandex.net
            return hostname.split('.')[1].lower()

        hostname = socket.gethostname()
        if is_qyp_hostname(hostname):
            return get_qyp_dc(hostname)

        # unknown dc is OK for local development
        return 'xxx'


DEFAULT = object()


class ResourceExplorerLazyProxy(object):
    def __init__(self):
        self.explorer = None

    def init(self):
        self.explorer = ResourceExplorer()
        return self.explorer

    @property
    def initialized(self):
        return bool(self.explorer)

    def __getattr__(self, item):
        if not self.initialized:
            raise Exception("ResourceExplorer is not initialized; You can't call '{}'".format(item))

        return getattr(self.explorer, item)


resource_explorer = ResourceExplorerLazyProxy()


def is_qyp_hostname(hostname):
    """
    :param hostname: e.g. rasp-teamcity-trusty-1.sas.yp-c.yandex.net
    :return:
    """
    return hostname.endswith('.yp-c.yandex.net')


def get_qyp_dc(hostname):
    return hostname.replace('.yp-c.yandex.net', '').split('.')[-1]
