# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import errno
import inspect
import os.path
import warnings
from datetime import datetime
from importlib import import_module
from shutil import rmtree

from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning


def remove_tmp_dir(temporary_directory):
    if os.path.exists(temporary_directory):
        rmtree(temporary_directory)


def get_relative_path(path, base_path):
    warnings.warn(
        "[2017-12-21] use os.path.relpath",
        RaspDeprecationWarning, stacklevel=2
    )
    return os.path.relpath(path, base_path)


def get_project_relative_path(path):
    from django.conf import settings

    return os.path.relpath(path, settings.PROJECT_PATH)


def is_project_path(path):
    from django.conf import settings

    return os.path.abspath(path).startswith(os.path.abspath(settings.PROJECT_PATH))


def get_class_script_path(cls):
    """
    Получем путь к файлу, содержащему класс, если не получилось, возвращаем имя модуля.
    """
    try:
        module = import_module(cls.__module__)
    except ImportError:
        return cls.__module__

    if hasattr(module, '__file__'):
        return get_project_relative_path(module.__file__)
    else:
        return cls.__module__


def get_my_caller_file(stack):
    caller_frame = stack[1]
    filename = caller_frame[1]
    return filename


def get_current_file_dir():
    stack = inspect.stack()
    caller_file_path = get_my_caller_file(stack)

    caller_file_name = os.path.basename(caller_file_path)

    name, _ext = os.path.splitext(caller_file_name)

    return os.path.dirname(os.path.normpath(os.path.abspath(caller_file_path)))


def get_current_file_data_dir():
    stack = inspect.stack()
    caller_file_path = get_my_caller_file(stack)

    caller_file_name = os.path.basename(caller_file_path)

    name, _ext = os.path.splitext(caller_file_name)

    dir_name = os.path.dirname(os.path.normpath(os.path.abspath(caller_file_path)))

    if name == '__init__':
        return os.path.join(dir_name, 'data')
    else:
        return os.path.join(dir_name, 'data', name)


def smart_split_ext(path):
    if path.endswith(('.xml.gz', '.tar.gz')):
        no_ext_path_1, ext_1 = os.path.splitext(path)
        no_ext_path, ext_2 = os.path.splitext(no_ext_path_1)
        return no_ext_path, ext_2 + ext_1

    return os.path.splitext(path)


def get_file_modify_dt(path):
    try:
        modify_timestamp = os.path.getmtime(path)
    except OSError as ex:
        if ex.errno == errno.ENOENT:  # no such file or directory
            return None
        raise

    return datetime.utcfromtimestamp(modify_timestamp)
