# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from collections import namedtuple
from itertools import islice
from math import acos, atan2, ceil, cos, degrees, radians, sin, sqrt

from travel.rasp.library.python.common23.utils.exceptions import SimpleUnicodeException


class GreatCircleCalculationError(SimpleUnicodeException):
    pass


EARTH_RADIUS_METRE = 6371009.0  # средний радиус Земли из https://en.wikipedia.org/wiki/Earth_radius#Mean_radius
EARTH_RADIUS_KILOMETRE = EARTH_RADIUS_METRE / 1000.0


def great_circle_angular_distance(point1, point2):
    u"""
    Вычисляет дистанцию в радианах между точками
    @param point1: Первая точка
    @param point2: Вторая точка
    У обоих станций точек быть аттрибуты longitude, latitude и они не должны
    быть None.
    """

    assert point1.longitude is not None and point1.latitude is not None
    assert point2.longitude is not None and point2.latitude is not None
    latitude_1_rad = radians(point1.latitude)
    longitude_1_rad = radians(point1.longitude)

    latitude_2_rad = radians(point2.latitude)
    longitude_2_rad = radians(point2.longitude)

    longitude_delta = longitude_2_rad - longitude_1_rad
    numerator = sqrt(
        (cos(latitude_2_rad) * sin(longitude_delta)) ** 2 +
        (
            cos(latitude_1_rad) * sin(latitude_2_rad) -
            sin(latitude_1_rad) * cos(latitude_2_rad) * cos(longitude_delta)
        ) ** 2
    )

    denominator = sin(latitude_1_rad) * sin(latitude_2_rad) + \
        cos(latitude_1_rad) * cos(latitude_2_rad) * cos(longitude_delta)

    return atan2(numerator, denominator)


def great_circle_distance_km(point1, point2):
    u"""
    Вычисляет дистанцию в километрах между точками
    @param point1: Первая точка
    @param point2: Вторая точка
    У обоих станций точек быть аттрибуты longitude, latitude и они не должны
    быть None.
    """

    if point1.longitude is None or point1.latitude is None:
        raise GreatCircleCalculationError(u"У первой точки не заполнены координаты")

    if point2.longitude is None or point2.latitude is None:
        raise GreatCircleCalculationError(u"У второй точки не заполнены координаты")

    return EARTH_RADIUS_KILOMETRE * great_circle_angular_distance(point1, point2)


def _clip_current(objects, center_lng, center_lat, span_lng, span_lat, now):
    # клиппинг по текущему положению

    left = (center_lng - span_lng / 2 + 180) % 360 - 180
    right = (center_lng + span_lng / 2 + 180) % 360 - 180

    bottom = center_lat - span_lat / 2
    top = center_lat + span_lat / 2

    for o in objects:
        # Положение в данный момент
        lng, lat = o.current_position(now)

        # Если не попадает во вьюпорт, продолжаем
        if left < right:
            if lng < left or lng > right:
                continue
        else:
            if lng > left and lng < right:
                continue

        if lat < bottom or lat > top:
            continue

        # Попадает во вьюпорт
        yield o


def center_span_zoom(request):
    try:
        center = request.GET['center']
        span = request.GET['span']

        center_lon, center_lat = [float(c) for c in center.split(',')]
        span_lon, span_lat = [float(c) for c in span.split(',')]

        center = center_lon, center_lat
        span = span_lon, span_lat
    except KeyError:
        center = None
        span = None

    try:
        zoom = int(float(request.GET['zoom']))
    except (KeyError, ValueError, TypeError):
        zoom = None

    return center, span, zoom


def clip(objects, center, span, radius=0, limit=None, now=None, zoom_radius=1, lon_field='lng', lat_field='lat'):
    center_lng, center_lat = center
    span_lng, span_lat = span

    if span_lng == 0 or span_lat == 0:
        return objects.none()

    span_lng *= zoom_radius
    span_lat *= zoom_radius

    # Фильтруем по горизонтали только если область не закольцована
    if span_lng < 360 - radius * 2:
        # Отношение площади вьюпорта к площади зоны поиска
        viewport_ratio = 1 + radius * 2 / span_lng

        left = (center_lng - span_lng / 2 - radius + 180) % 360 - 180
        right = (center_lng + span_lng / 2 + radius + 180) % 360 - 180

        # log.debug("%f, %f" % (left, right))

        if left < right:  # нет перехода области обзора через границу 180 градусов
            objects = objects.filter(**{"%s__range" % lon_field: [left, right]})
        else:  # переход через границу 180 градусов, right < left
            # Выкидываем те рейсы, которые не попадают в область
            objects = objects.exclude(**{"%s__range" % lon_field: [right, left]})

        # Фильтруем по вертикали если охвачено менее 160 градусов
        if span_lat < 160:

            viewport_ratio *= 1 + radius * 2 / span_lat

            bottom = center_lat - span_lat / 2 - radius
            top = center_lat + span_lat / 2 + radius
            objects = objects.filter(**{"%s__range" % lat_field: [bottom, top]})

        if limit:
            # Расширяем лимит, чтобы в реальный вьюпорт попало около limit
            # объектов
            query_limit = max(0, min(int(ceil(limit * viewport_ratio)), 300))

            objects = _clip_current(objects.order_by('id')[:query_limit], center_lng, center_lat,
                                    span_lng, span_lat, now)

            return islice(objects, limit)

    return objects


def clip_args(request, radius):
    try:
        center = request.GET['center']
        span = request.GET['span']
    except KeyError:
        return {}

    try:
        center_lng, center_lat = [float(c) for c in center.split(',')]
        span_lng, span_lat, = [float(c) for c in span.split(',')]
    except ValueError:
        return {}

    args = {}

    # Фильтруем по горизонтали только если область не закольцована
    if span_lng < 360 - radius * 2:
        span_lng += radius * 2
        span_lat += radius * 2

        left = (center_lng - span_lng / 2 + 180) % 360 - 180
        right = (center_lng + span_lng / 2 + 180) % 360 - 180

        args['left_right'] = left, right

        # Фильтруем по вертикали если охвачено менее 160 градусов
        if span_lat < 160:
            bottom = center_lat - span_lat / 2
            top = center_lat + span_lat / 2

            args['bottom_top'] = bottom, top

    return args


class PolylineClipper(object):
    """Обрезалка полилиний во вьюпорт"""

    # Квадраты:
    # 012
    # 345
    # 678

    # В матрице по вертикали и горизонтали - квадраты точек
    # 1 - прямая, проходящая через точки, возможно пересекает 4,
    # 0 - не пересекает

    direct = [
        [0, 0, 0, 0, 1, 1, 0, 1, 1],
        [0, 0, 0, 1, 1, 1, 1, 1, 1],
        [0, 0, 0, 1, 1, 0, 1, 1, 0],
        [0, 1, 1, 0, 1, 1, 0, 1, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1],
        [1, 1, 0, 1, 1, 0, 1, 1, 0],
        [0, 1, 1, 0, 1, 1, 0, 0, 0],
        [1, 1, 1, 1, 1, 1, 0, 0, 0],
        [1, 1, 0, 1, 1, 0, 0, 0, 0],
    ]

    # То-же самое, но нас интересуют квадраты 3 и 5
    # Для цилиндрических вьюпортов

    # 012
    # 345
    # 678

    inverted = [
        [0, 0, 0, 1, 1, 1, 1, 1, 1],
        [0, 0, 0, 1, 0, 1, 1, 0, 1],
        [0, 0, 0, 1, 1, 1, 1, 1, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1],
        [1, 0, 1, 1, 0, 1, 1, 0, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1],
        [1, 1, 1, 1, 1, 1, 0, 0, 0],
        [1, 0, 1, 1, 0, 1, 0, 0, 0],
        [1, 1, 1, 1, 1, 1, 0, 0, 0],
    ]

    @classmethod
    def from_request(cls, request, radius):
        return cls(**clip_args(request, radius))

    def __init__(self, left_right=None, bottom_top=None):
        self.left_right = left_right
        self.bottom_top = bottom_top

        self.solutions = self.direct

        if self.left_right and self.left_right[0] >= self.left_right[1]:
            self.solutions = self.inverted

    def third(self, c, bounds):
        if not bounds:
            return 1

        if c < bounds[0]:
            return 0

        if c > bounds[1]:
            return 2

        return 1

    def test(self, c1, c2):
        r1 = self.third(c1[0], self.left_right) + 3 * self.third(c1[1], self.bottom_top)
        r2 = self.third(c2[0], self.left_right) + 3 * self.third(c2[1], self.bottom_top)

        return self.solutions[r1][r2]


def get_square_around_point(point, distance_km):
    """
    Для слишком больших дистанций вычисления могут быть не верными
    """
    half_of_side = distance_km / 2.
    latitude_delta_rad = half_of_side / EARTH_RADIUS_KILOMETRE

    # Близко к полюсам расчеты не ведем
    etalon_lat = point.latitude
    if etalon_lat > get_square_around_point.MAX_LATITUDE:
        etalon_lat = get_square_around_point.MAX_LATITUDE

    elif etalon_lat < -get_square_around_point.MAX_LATITUDE:
        etalon_lat = -get_square_around_point.MAX_LATITUDE

    etalon_lat_rad = radians(etalon_lat)

    longitude_delta_rad = acos(
        (cos(latitude_delta_rad) - sin(etalon_lat_rad)**2) /
        cos(etalon_lat_rad)**2
    )

    lat_delta_deg = degrees(abs(latitude_delta_rad))
    lng_delta_deg = degrees(abs(longitude_delta_rad))

    lat_top = point.latitude - lat_delta_deg
    lat_bottom = point.latitude + lat_delta_deg

    lng_left = point.longitude - lng_delta_deg
    lng_right = point.longitude + lng_delta_deg

    return (lat_top, lng_left), (lat_bottom, lng_right)


get_square_around_point.MAX_LATITUDE = 80


class GeoPoint(namedtuple('GeoPoint', 'latitude longitude')):

    def expand_square(self, distance):
        return tuple(
            GeoPoint(*coords)
            for coords in get_square_around_point(self, distance * 2)
        )

    def distance(self, other):
        return great_circle_distance_km(self, other)
