# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from builtins import next
from collections import OrderedDict
from itertools import tee, groupby, product, islice
from operator import itemgetter

from six.moves import filterfalse, zip as izip, map as imap


def unique_everseen(iterable, key=None):
    """
    List unique elements, preserving order. Remember all elements ever seen.
    unique_everseen('AAAABBBCCDAABBB') --> A B C D
    unique_everseen('ABBCcAD', str.lower) --> A B C D
    """

    seen = set()
    seen_add = seen.add
    if key is None:
        for element in filterfalse(seen.__contains__, iterable):
            seen_add(element)
            yield element
    else:
        for element in iterable:
            k = key(element)
            if k not in seen:
                seen_add(k)
                yield element


def unique_justseen(iterable, key=None):
    """
    List unique elements, preserving order. Remember only the element just seen.
    unique_justseen('AAAABBBCCDAABBB') --> A B C D A B
    unique_justseen('ABBCcAD', str.lower) --> A B C A D
    """

    return imap(lambda i: next(i), imap(itemgetter(1), groupby(iterable, key)))


def pairwise(iterable):
    """
    s -> (s0,s1), (s1,s2), (s2, s3), ...
    """

    a, b = tee(iterable)
    try:
        next(b)
    except StopIteration:
        pass
    return izip(a, b)


def product_by_key(spec):
    """
    Generate list of dicts with keys from `spec` and values from product of `spec` values.
    >>> product_by_key({'foo': [1, 2], 'bar': ['a', 'b']})
    [{'foo': 1, 'bar': 'a'}, {'foo': 1, 'bar': 'b'}, {'foo': 2, 'bar': 'a'}, {'foo': 2, 'bar': 'b'}]
    """
    spec = OrderedDict(spec.items())
    return [
        dict(izip(list(spec.keys()), v)) for v in product(*list(spec.values()))
    ]


def chunker(iterable, chunk_size):
    iterator = iter(iterable)
    while True:
        chunk = list(islice(iterator, chunk_size))
        if chunk:
            yield chunk
        else:
            break
