# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import bytes
from io import BytesIO

from django.core.files.base import File
from django.utils.encoding import smart_bytes, smart_text


class MemoryFile(File):
    def __init__(self, name, content_type, content):
        name = smart_text(name)
        self.content_type = content_type
        super(MemoryFile, self).__init__(BytesIO(content), name)
        self.size = len(content)

    DELIMITER = b'\r\n'

    def to_binary_string(self):
        self.open()
        data = self.read()

        parts = [smart_bytes(self.name, strings_only=True),
                 smart_bytes(self.content_type, strings_only=True),
                 data]

        self.open()

        return self.DELIMITER.join(parts)

    def open(self, mode=None):
        self.seek(0)

    def copy(self):
        self.seek(0)
        content = self.read()
        self.seek(0)
        return MemoryFile(self.name, self.content_type, content)

    @classmethod
    def from_file_like_object_string(cls, file_object, content_type):
        return cls(file_object.name, content_type, file_object.read())

    @classmethod
    def from_binary_string(cls, binary_string):
        cls.check_binary_string(binary_string)

        return cls(*binary_string.split(cls.DELIMITER, 2))

    @classmethod
    def check_binary_string(cls, binary_string):
        if not isinstance(binary_string, bytes):
            raise TypeError('bytes type must be provided not %s' % type(binary_string))

        if binary_string.count(cls.DELIMITER) < 2:
            raise ValueError('input string does not contain delimiters. Conversion is not posible.')

    def __unicode__(self):
        return self.name
