# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from future import standard_library
standard_library.install_aliases()
from builtins import map
from os import getpid

from functools import wraps
from six.moves._thread import get_ident
from time import time

stacks = {}


ENABLE_TRACER = False


class Frame(object):
    def __init__(self, name, attrs):
        self.name = name
        self.children = []
        self.attrs = attrs
        self.pid = None

    def enter(self):
        self.pid = getpid()

        enter = self.attrs['enter'] = {}

        enter['size'], enter['resident'] = get_size_resident(self.pid)
        enter['time'] = time()

    def leave(self):
        exit_ = self.attrs['exit'] = {'time': time()}

        if self.pid is not None:
            self.attrs['pid'] = self.pid
            exit_['size'], exit_['resident'] = get_size_resident(self.pid)

    def add(self, frame):
        self.children.append(frame)

    def set_attr(self, name, value):
        self.attrs[name] = value

    def __json__(self):
        o = {
            'name': self.name,
            'children': self.children,
        }

        o.update(self.attrs)

        return o


class Manager(object):
    def __init__(self, name, **attrs):
        self.name = name
        self.frame = Frame(name, attrs)

    def __enter__(self):
        try:
            if not ENABLE_TRACER:
                return
        except ImportError:
            return

        stack = stacks.setdefault(get_ident(), [])

        if stack:
            stack[-1].add(self.frame)

        stack.append(self.frame)

        self.frame.enter()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.frame.leave()

        ident = get_ident()

        try:
            stack = stacks[ident]
        except KeyError:
            return

        top_frame = stack.pop()

        while top_frame != self.frame:
            top_frame = stack.pop()

        if not stack:
            del stacks[ident]

        return False

    def leave(self):
        self.__exit__(None, None, None)


def _wrap(f, name):
    @wraps(f)
    def wrapper(*args, **kwargs):
        with Manager(name):
            return f(*args, **kwargs)

    return wrapper


def wrap(arg):
    if hasattr(arg, '__call__'):
        return _wrap(arg, arg.__name__)

    def decorator(f):
        return _wrap(f, arg)

    return decorator


def enter(name):
    manager = Manager(name)

    manager.__enter__()

    return manager


def start_thread(thread):
    manager = Manager('start thread')

    with manager:
        thread_manager = Manager('thread')

        thread.frame = thread_manager.frame

        thread_run = thread.run

        def wrapper():
            with thread_manager:
                thread.frame.set_attr('ident', get_ident())
                thread_run()

        thread.run = wrapper

        thread.start()

        manager.frame.add(thread.frame)


def join_thread(thread):
    manager = Manager('join thread', thread=thread.ident)

    with manager:
        thread.join()
        manager.frame.add(thread.frame)


def set_bottom_frame_attr(name, value):
    stack = stacks.get(get_ident())

    if stack:
        bottom_frame = stack[0]

        bottom_frame.set_attr(name, value)


def get_size_resident(pid):
    # size       total program size
    #            (same as VmSize in /proc/[pid]/status)
    # resident   resident set size
    #            (same as VmRSS in /proc/[pid]/status)
    # share      shared pages (from shared mappings)
    # text       text (code)
    # lib        library (unused in Linux 2.6)
    # data       data + stack
    # dt         dirty pages (unused in Linux 2.6)

    with open('/proc/%d/statm' % pid) as f:
        line = f.readline()

    fields = line.strip().split()

    assert len(fields) == 7

    size, resident, share, text, lid, data, dt = map(int, fields)

    return size, resident


def tag(tag):
    set_bottom_frame_attr('tag', tag)
