# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import map
import logging
import os

from django.conf import settings
from enum import IntEnum

import geobase6

from travel.rasp.library.python.common23.logging.logging_call_func import logging_call_func


log = logging.getLogger(__name__)


RASP_GEOBASE_LOG_CALLS = os.getenv('RASP_GEOBASE_LOG_CALLS', False)
RASP_GEOBASE_LAZY_LOAD = os.getenv('RASP_GEOBASE_LAZY_LOAD', False)


class GeobaseRegionType(IntEnum):
    HIDDEN = -1  # Скрытый регион.
    OTHER = 0  # Прочие регионы.
    CONTINENT = 1  # Континент.
    REGION = 2  # Регион.
    COUNTRY = 3  # Страна.
    FEDERAL_OKRUG = 4  # Федеральный округ
    FEDERAL_SUBJECT = 5  # Субъект федерации.
    CITY = 6  # Город.
    VILLAGE = 7  # Село.
    CITY_DISTRICT = 8  # Район города.
    UNDEGROUND_STATION = 9  # Станция метро.
    FEDERAL_SUBJECT_DISTRICT = 10  # Район субъекта федерации.
    AIRPORT = 11  # Аэропорт.
    # Заморская территория. Заморские территории считаются принадлежащими части света,
    # в которой они расположены географически, а не их реальный владелец. Примеры заморских территорий:
    # Бермудские и Каймановы острова, Ангилья, Гренландия и пр.
    OVERSEAS_TERRITORY = 12
    CITY_DISTRICT_LEVEL_2 = 13  # Район города второго уровня.
    MONORAIL_STATION = 14  # Станция монорельса.
    VILLAGE_SETTLEMENT = 15  # Сельское поселение.


class _GeoRegion(object):
    pass


def wrap_geo_dict(geo_dict):
    geo_region = _GeoRegion()
    geo_region.__dict__.update(geo_dict)
    geo_region.ename = geo_region.en_name
    geo_region.timezone = geo_region.tzname
    geo_region.chief_region = geo_region.capital_id

    return geo_region


class _Geobase6Lookup(object):
    def __init__(self, impl, geobase_lib):
        self._impl = impl
        self._geobase = geobase_lib

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def __getattr__(self, item):
        return getattr(self._impl, item)

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def linguistics(self, geo_id, lang):
        return self._impl.get_linguistics(geo_id, lang)

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def regions_by_type(self, region_type):
        return list(map(wrap_geo_dict, self._impl.get_regions_by_type(region_type)))

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def region_by_id(self, geo_id):
        return wrap_geo_dict(self._impl.get_region_by_id(geo_id))

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def find_country_id(self, geo_id, national_version):
        return self._impl.get_country_id(geo_id, national_version)

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def children(self, geo_id):
        return self._impl.get_children_ids(geo_id)

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def parents(self, geo_id):
        return self._impl.get_parents_ids(geo_id)

    @logging_call_func(enabled=RASP_GEOBASE_LOG_CALLS)
    def calculate_points_distance(self, *args):
        return self._geobase.calculate_distance(*args)


def get_geobase_lookup():
    log.info('Loading geobase')
    geobase_data_path = os.getenv('RASP_GEOBASE_DATA_PATH', settings.GEOBASE_DATA_PATH)
    return _Geobase6Lookup(geobase6.Lookup(geobase_data_path), geobase6)


class GeobazeLazy(object):
    def __init__(self):
        self.geobase = None

    def __getattr__(self, item):
        if self.geobase is None:
            self.geobase = get_geobase_lookup()

        return getattr(self.geobase, item)


if RASP_GEOBASE_LAZY_LOAD:
    geobase = GeobazeLazy()
else:
    geobase = get_geobase_lookup()
