# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import sys
from threading import Thread, Lock, Event

import six
from django.conf import settings
from django.db import connection

from travel.rasp.library.python.common23.date import environment
from travel.rasp.library.python.common23.utils import tracer


class SimpleThread(Thread):
    """ Вспомогательный поток для функции run_in_thread """

    # Общий Lock на обновление входных параметров.
    # Предполагается, что функция будет использовать лок.
    _lock = Lock()

    def __init__(self, function, args, kwargs, lock_for_update, request=None):
        self._result = None
        self.request = request
        self.function = function
        self.args = args
        self.kwargs = kwargs
        self.lock_for_update = lock_for_update
        self._exc_info = None
        super(SimpleThread, self).__init__()

    def run(self):
        log = logging.getLogger(__name__)
        environment.activate_request(self.request)
        try:
            if self.lock_for_update:
                self.kwargs.update({'lock': self._lock})
            try:
                self._result = self.function(*self.args, **self.kwargs)
            finally:
                # Закрываем коннекшн, если он создался для треда
                self.close_connection()
        except Exception:
            self._exc_info = sys.exc_info()
            log.exception(u'Проблема запуска потока')

    def get_result(self):
        if self._exc_info is not None:
            six.reraise(self._exc_info[0], self._exc_info[1], self._exc_info[2])

        return self._result

    def close_connection(self):
        connection.close()


class FakeThread(SimpleThread):
    def start(self):
        self.run()

    def join(self, *args, **kwargs):
        pass

    def close_connection(self):
        pass


def run_in_thread(function, args=tuple(), kwargs=dict(), lock_for_update=False):
    """ Запускает функцию в потоке, возвращает объект потока.
    После завершения работы потока результаты работы функции можно получить с
    помощью метода потока get_result(),
    lock_for_update - параметр позволяющий вызвать функцию с
    дополнительным именованым параметром lock, lock создается один на программу.
    (Точнее на инстанс класса SimpleThread)
    Если в ходе выполнения потока возникли исключения,
    функция при вызове get_result() кидает Exception со стек трейсом.
    Так же стек трейс пишется в лог rasp.common
    """

    request = environment.get_request()

    thread = SimpleThread(function, args, kwargs, lock_for_update, request)
    if settings.DEBUG:
        # Не стартовать ничего в тредах (это съедает sql-запросы)
        thread = FakeThread(function, args, kwargs, lock_for_update, request)

    tracer.start_thread(thread)
    return thread


class Ticker(Thread):
    """
    Тред, выполняющий код через заданный промежуток времени. Может быть остановлен через .stop()

    class MyTicker(Ticker):
        def tick(self):
            print 123

    ticker = MyTicker(interval=1)
    ticker.start()
    sleep(10)
    ticker.stop()
    ticker.join()
    """

    def __init__(self, interval, group=None, target=None, name=None, daemon=True, args=(), kwargs=None, verbose=None):
        super(Ticker, self).__init__(group=group, name=name)

        self.target = target
        self.daemon = daemon
        self.args = args or ()
        self.kwargs = kwargs or {}

        self.interval = interval
        self.stop_event = Event()

    def run(self):
        func = self.target if self.target else self.tick
        while not self.stop_event.wait(self.interval):
            func(*self.args, **self.kwargs)

    def tick(self, *args, **kwargs):
        raise NotImplementedError

    def stop(self):
        self.stop_event.set()
