# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from builtins import next
import json
import logging
import os
import re
from textwrap import dedent

import six
from babel.messages.plurals import get_plural
from django.conf import settings
from django.utils import translation
from lxml import etree


log = logging.getLogger(__name__)


_global_keyset = {}
_keysets = {}
_keysets_stats = {}


def _reload_keysets_no_arcadia():
    changed = False

    for keyset, attrs in settings.XGETTEXT_KEYSETS.items():
        path = os.path.join(settings.PROJECT_PATH, attrs['filename'])

        try:
            stat = os.stat(path)
        except:
            stat = None

        keyset_stat = _keysets_stats.get(keyset)

        if stat is None and keyset_stat is None:
            continue

        if stat is not None and keyset_stat is not None:
            if all(getattr(stat, attr) == getattr(keyset_stat, attr)
                   for attr in ['st_size', 'st_mtime']):
                continue

        _keysets_stats[keyset] = stat

        keyset_data = {}

        try:
            with open(path) as f:
                keyset_data = json.load(f)
        except:
            pass

        _keysets[keyset] = keyset_data
        _keysets_stats[keyset] = stat

        changed = True

    return changed


def _reload_keysets_arcadia():
    changed = False

    for keyset, attrs in settings.XGETTEXT_KEYSETS.items():
        file_path = 'resfs/file/' + attrs['filename']

        if keyset in _keysets:
            continue

        try:
            data = next(resource.itervalues(file_path))
        except StopIteration:
            log.warning('Keyset not found %s', file_path)
            continue

        _keysets[keyset] = json.loads(data)
        changed = True

    return changed


# change _reload_keysets for Arcadia
try:
    from library.python import resource
except ImportError:
    _reload_keysets = _reload_keysets_no_arcadia
    resource = None
else:
    _reload_keysets = _reload_keysets_arcadia


def get_keyset():
    global _global_keyset

    changed = _reload_keysets()

    if changed:
        for keyset in _keysets.values():
            _global_keyset.update(keyset)

    return _global_keyset


KEY_RE = re.compile(r'[\n ]+', re.U)


def form_key(form):
    return KEY_RE.sub(' ', form).strip()


def _gettext(form, key=None, lang=None):
    lang = lang or translation.get_language()

    form = dedent(form).strip()

    if not key:
        key = form_key(form)

    try:
        data = get_keyset()[key]
    except KeyError:
        return form

    assert not data["info"]["is_plural"]

    try:
        key_translation = data["translations"][lang]
    except KeyError:
        return form

    translated_form = key_translation["form"]

    if translated_form:
        return translated_form

    return form


def _ngettext(n, *forms, **kwargs):
    lang = translation.get_language()

    forms = [dedent(form).strip() for form in forms]

    num_plurals, func = get_plural_func(settings.BASE_LANGUAGE)

    assert len(forms) == num_plurals

    base_form = forms[func(n)]

    key = kwargs.pop('key', None)

    if kwargs:
        raise ValueError("Only allowed keyword argument is key")

    if not key:
        key = form_key(forms[0])

    try:
        data = get_keyset()[key]
    except KeyError:
        return base_form

    assert data["info"]["is_plural"]

    try:
        key_translation = data["translations"][lang]
    except KeyError:
        return base_form

    num_plurals, func = get_plural_func(lang)

    form_n = func(n)

    translated_form = key_translation["form%d" % (form_n + 1)]

    if translated_form:
        return translated_form

    return base_form


def marktrans(form):
    if getattr(translation, 'marktrans', False):
        if isinstance(form, six.string_types):
            return '[trans]%s[/trans]' % form

        return ['[trans]', form, '[/trans]']

    return form


def markdbtrans(form):
    if getattr(translation, 'marktrans', False):
        if isinstance(form, six.string_types):
            return '[dbtrans]%s[/dbtrans]' % form

        return ['[dbtrans]', form, '[/dbtrans]']

    return form


def gettext(form, **kwargs):
    return marktrans(_gettext(form, **kwargs))


class LazyForm(object):
    def __init__(self, form):
        self.form = form


def mark_gettext(form, key=None):
    """
    Помечает form для выгрузки в tanker и возвращает ключ для получения перевода с помощью dynamic_gettext.

    Пример использования:

        ARRIVAL_TEXT_KEY = mark_gettext(u'прибытие')
        ...
        dynamic_gettext(ARRIVAL_TEXT_KEY)

    Аналогично с key:

        DEPARTURE_TEXT_KEY = mark_gettext(u'отправление', key='departure_text')
        ...
        dynamic_gettext(DEPARTURE_TEXT_KEY)

    """
    return key or form


def dynamic_gettext(form, **kwargs):
    """
    Производит локализацию константы ранее помеченной методом mark_gettext.
    """
    return marktrans(_gettext(form, **kwargs))


_plural_func_cache = {}


def get_plural_func(lang):
    import hemi

    try:
        return _plural_func_cache[lang]
    except KeyError:
        pass

    plural = get_plural(lang)

    source = "(function(n) { return %s })" % plural.plural_expr

    func = hemi.Context().eval(source)

    _plural_func_cache[lang] = plural.num_plurals, func

    return _plural_func_cache[lang]


def ngettext(n, *forms, **kwargs):
    return marktrans(_ngettext(n, *forms, **kwargs))


DOCTYPE_STRING = """<!DOCTYPE html [
    <!ENTITY nbsp   "&#160;"> <!-- no-break space = non-breaking space, U+00A0 ISOnum -->
    <!ENTITY laquo  "&#171;"> <!-- left-pointing double angle quotation mark = left pointing guillemet, U+00AB ISOnum -->
    <!ENTITY raquo  "&#187;"> <!-- right-pointing double angle quotation mark = right pointing guillemet, U+00BB ISOnum -->
    <!ENTITY mdash   "&#8212;"> <!-- em dash, U+2014 ISOpub -->
]>"""  # noqa: E501
XML_HEADER = '<?xml version="1.0" standalone="yes"?>'


class XFormatParseError(Exception):
    pass


def xgettexttree(form):
    xml = XML_HEADER + DOCTYPE_STRING + '<xml>' + form + '</xml>'

    try:
        tree = etree.fromstring(xml)
    except Exception as e:
        raise XFormatParseError("Error parsing form %r: %r" % (form, e))

    return tree


def xformat(form, **kwargs):
    root = xgettexttree(form)

    def process(element):
        content = []

        if element.text:
            content.append(element.text)

        for child in element:
            child_content = process(child)

            handler_name = child.tag.replace('-', '_')

            if handler_name in kwargs:
                handler = kwargs[handler_name]

                if child_content:
                    child_content = handler(child_content, **child.attrib)

                elif child.attrib or callable(handler):
                    child_content = handler(**child.attrib)

                else:
                    child_content = handler

            else:
                elem = etree.Element(child.tag, child.attrib)

                elem.text = ''

                elemxml = etree.tostring(elem, encoding='unicode')

                split_index = elemxml.index('><') + 1

                child_content = [
                    elemxml[:split_index],
                    child_content,
                    elemxml[split_index:]
                ]

            content.append(child_content)

            if child.tail:
                content.append(child.tail)

        return content

    return process(root)


def tformat(form, **handlers):
    def make_tree_handler(handler):
        def tree_handler(*args, **kwargs):
            return handler(stringify(args[0]), **kwargs) if args else handler(**kwargs)
        return tree_handler

    for tag, handler in handlers.items():
        if callable(handler):
            handlers[tag] = make_tree_handler(handler)

    return stringify(xformat(form, **handlers))


def stringify(bemjson):
    def iter_tree(tree):
        for branch in tree:
            if isinstance(branch, list):
                for leaf in iter_tree(branch):
                    yield leaf
            else:
                yield six.text_type(branch)

    return u''.join(iter_tree(bemjson))


def xgettext(form, **kwargs):
    """
    Возвращает BEM-json содержащий перевод для form

    Пример:
    tgettext("Это '<word />' слово", **{ 'word': 'слово' })

    Результат:
    ["Это '", "слово", "' слово"]
    """

    key = kwargs.pop('key', None)
    lang = kwargs.pop('lang', None)

    form = _gettext(form, key=key, lang=lang)

    return marktrans(xformat(form, **kwargs))


def tgettext(form, **kwargs):
    """
    Возвращает строку содержащую перевод для form

    Пример:
    tgettext("Это '<word />' слово", **{ 'word': 'слово' })

    Результат:
    Это 'слово' слово
    """

    return stringify(xgettext(form, **kwargs))


def xngettext(n, *forms, **kwargs):
    """
    Возвращает BEM-json содержащий перевод для forms с учетом plural-формы

    Пример:
    xngettext(1, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["1", " слово"]

    Пример:
    xngettext(2, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["2", " слова"]

    Пример:
    xngettext(25, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["25", " слов"]
    """

    form = _ngettext(n, *forms, key=kwargs.pop('key', None))

    kwargs["n"] = n

    return marktrans(xformat(form, **kwargs))


def tngettext(n, *forms, **kwargs):
    """
    Возвращает строку содержащую перевод для forms с учетом plural-формы

    Пример:
    tngettext(1, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "1 слово"

    Пример:
    tngettext(2, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "2 слова"

    Пример:
    tngettext(25, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "25 слов"
    """

    return stringify(xngettext(n, *forms, **kwargs))
