# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import warnings
from itertools import chain

from common.models.geo import Station, StationTerminal
from common.utils.i18n import CollateUnicode
from common.utils.iterrecipes import unique_justseen

from travel.rasp.library.python.common23.models.utils import fetch_children
from travel.rasp.library.python.common23.utils.warnings import RaspDeprecationWarning


def get_stations_by_type(settlement, plane_only=False, use_water=False):
    """ TODO используется только на старом стеке, отпилить после выкатки станции
        Возвращает станции всех типов транспорта для данного города
        Вход: город - объект Settlement
        Выход: {'plane': [st1, st2], 'train': [st3, st4]}
    """
    from common.models.transport import TransportType

    if not use_water:
        warnings.warn('[2015-11-30] use_water must be True', RaspDeprecationWarning, stacklevel=2)

    t_types = TransportType.objects.in_bulk_cached()
    result = {
        t_types[t_type_code].code: {'type': t_types[t_type_code], 'stations': [], 'related': []}
        for t_type_code in t_types.keys()
    }

    qs = Station.objects.filter(hidden=False, majority__id__lt=3)

    stations = list(chain(qs.filter(settlement=settlement),
                          qs.filter(station2settlement__settlement=settlement)))

    stations.sort(key=lambda s: (s.majority_id, CollateUnicode(s.get_clean_title().strip('"')), s.id))

    terminals = fetch_children(stations, StationTerminal.objects, 'station')

    # Разносим станции по типам ТС
    for st in unique_justseen(stations, key=lambda s: s.id):
        ttype_id = st.t_type_id
        if plane_only and ttype_id != TransportType.PLANE_ID:
            continue

        ttype_code = st.t_type.code
        if ttype_id in TransportType.WATER_TTYPE_IDS:
            ttype_code = 'water'

        result[ttype_code]['stations'].append((st, terminals.getlist(st.id)))

    related = (settlement.related_stations.filter(station__hidden=False, station__majority__id__lt=3)
               .order_by('station__majority__id', 'station__popular_title', 'station__title'))

    for related_station in related:
        station = related_station.station

        ttype_id = station.t_type_id
        if plane_only and ttype_id != TransportType.PLANE_ID:
            continue

        ttype_code = station.t_type.code
        if ttype_id in TransportType.WATER_TTYPE_IDS:
            ttype_code = 'water'

        result[ttype_code]['related'].append((station, terminals.getlist(station.id)))

    if not use_water and 'water' in result:
        result['sea'] = result.pop('water')

    if use_water and 'sea' in result:
        del result['sea']

    if use_water and 'river' in result:
        del result['river']

    return result
