# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import typing

from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info.containers import ClusterInfo
from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info import MdbAPI

from travel.rasp.library.python.db.mysql import utils
from travel.rasp.library.python.db.cluster import ClusterBase, DbInstance, LOW_INSTANCE_PRIORITY, ClusterPeriodicUpdateMixin, ClusterException
from travel.rasp.library.python.db.replica_health import ReplicaHealth


log = logging.getLogger(__name__)


ANY_REPLICA_HOSTNAME_TEMPLATE = 'c-{}.ro.db.yandex.net'
ANY_MASTER_HOSTNAME_TEMPLATE = 'c-{}.rw.db.yandex.net'


class ClusterMdb(ClusterPeriodicUpdateMixin, ClusterBase):
    def __init__(
        self,
        mdb_client,
        cluster_id,
        mdb_api_call_enabled=True,
        check_master_on_each_connect=False,
        fallback_replicas=None,
        fallback_master=None,
        *args, **kwargs
    ):
        """
        :param mdb_client: клиент MdbAPI
        :param cluster_id: id кластера в терминах MDB
        :param mdb_api_call_enabled: можно ли делать запросы в АПИ MDB.
        :param check_master_on_each_connect: перед получением соединения проверять, какой инстанс является мастером сейчас
        :param fallback_replicas: список хостов для фолбэка, если АПИ не доступно
        :param fallback_master: мастер для фолбэка, если АПИ не доступно

        Про mdb_api_call_enabled.
        Разработчики MDB не рекомендуют использовать АПИ для получения хостов, а вместо этого прописывать их в конфигах, т.к.:
        - хостнеймы инстансов не меняются
        - новые хосты добавляются редко
        - ручка АПИ по получению хостов кластера очень тяжелая и не готова к нагрузкам
        Поэтому во многих случаях стоит хранить список хостов в конфигурации и передавать его через fallback, а в АПИ не ходить.
        """
        super(ClusterMdb, self).__init__(*args, **kwargs)

        self.mdb_client = mdb_client  # type: MdbAPI
        self.mdb_api_call_enabled = mdb_api_call_enabled
        self.cluster_id = cluster_id  # type: str  # MDB cluster id
        self.check_master_on_each_connect = check_master_on_each_connect

        self.mdb_cluster_info = None  # type: typing.Optional[ClusterInfo]

        # fallback-хосты используются, если mdb-клиент не смог получить хосты из АПИ или из кэша
        self.fallback_master = None
        self.fallback_replicas = None
        self.replicas_health = {}

        if fallback_master:
            self.set_fallback_hosts(fallback_master, fallback_replicas)

    def set_fallback_hosts(self, fallback_master, fallback_replicas):
        self.fallback_master = fallback_master
        self.fallback_replicas = fallback_replicas or []
        self.mdb_client.add_default_cluster_info(self.cluster_id, master_hostname=self.fallback_master, hostnames=self.fallback_replicas)
        log.info("Set fallback hosts for cluster %s to %s, %s", self.cluster_id, self.fallback_master, self.fallback_replicas)

    def set_replicas_health(self, replicas_health):
        self.replicas_health = replicas_health or {}
        log.info("Set replicas health for cluster %s %s", self.cluster_id, self.replicas_health)

    def get_mdb_cluster_info(self):
        if self.mdb_api_call_enabled:
            return self.mdb_client.get_cluster_info(self.cluster_id)
        else:
            return self.mdb_client.get_default_cluster_info(self.cluster_id)

    def get_actual_instances_list(self):
        # type: () -> typing.List[DbInstance]
        instances = []
        try:
            self.mdb_cluster_info = self.get_mdb_cluster_info()
        except Exception:
            self.log.exception("Can't update mdb_cluster_info %s", self.cluster_id)
        else:
            instances += [
                DbInstance(
                    host=mdb_inst.hostname,
                    is_master=mdb_inst.is_master,
                    dc=mdb_inst.dc,
                    health=self.replicas_health.get(mdb_inst.hostname, ReplicaHealth.ALIVE),
                    priority=0,
                )
                for mdb_inst in self.mdb_cluster_info.instances
            ]

        default_replica = DbInstance(
            host=ANY_REPLICA_HOSTNAME_TEMPLATE.format(self.cluster_id),
            is_master=False, dc=None, priority=LOW_INSTANCE_PRIORITY
        )
        instances.append(default_replica)

        no_master = all(not inst.is_master for inst in instances)
        if no_master:
            default_master = DbInstance(
                host=ANY_MASTER_HOSTNAME_TEMPLATE.format(self.cluster_id),
                is_master=True, dc=None, priority=LOW_INSTANCE_PRIORITY
            )
            instances.append(default_master)

        return instances

    def get_connection(self, current_dc=None):
        if self.check_master_on_each_connect:
            master_host = self.find_master_host()
            if master_host:
                self.set_master_instance_by_host(master_host)

        return super(ClusterMdb, self).get_connection(current_dc)

    def find_master_host(self):
        # type: () -> typing.AnyStr
        """
        Возвращает хост инстанса, который является мастером. Реализация зависит от типа базы
        """
        raise NotImplementedError

    def set_master_instance_by_host(self, new_master_host):
        old_master, new_master = None, None
        for inst in self.instances:
            if inst.host == new_master_host:
                new_master = inst

            if inst.is_master:
                old_master = inst

        if not new_master:
            raise ClusterException("Can't switch master to host {}: no such instance".format(new_master_host))

        if old_master:
            old_master.is_master = False

        new_master.is_master = True

    def __repr__(self):
        return "ClusterMdb: {}".format(self.cluster_id)


class ClusterMdbMysql(ClusterMdb):
    def find_master_host(self):
        # type: () -> typing.AnyStr
        for inst in self.instances:
            try:
                conn = self.get_connection_to_instance(inst)
            except Exception as ex:
                self.log.warning("Can't find master using instance %s: %s", inst, repr(ex))
            else:
                master_host = utils.get_master_host_from_conn(conn)
                if master_host:
                    return master_host
                else:  # master not found from instance -> this instance is master
                    return inst.host
