# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import time
import typing

import six


log = logging.getLogger(__name__)


class BadReplicasError(Exception):
    pass


class ReplicaSyncChecker(object):
    def __init__(self, hosts, conn_getter, is_synced, max_try_count=3, try_delay=30, wait_for_seconds=3600):
        # type: (typing.List[six.string_types], typing.Callable[[six.string_types], typing.Any], typing.Callable[[typing.Any], bool], int, int, int) -> None
        """
        :param hosts:
        :param conn_getter:
        :param is_synced: callable to check if host is in required state; arg is established connection
        :param max_try_count:
        :param try_delay:
        :param wait_for_seconds:
        """
        self.hosts = hosts
        self.conn_getter = conn_getter
        self.is_synced = is_synced

        self.report_interval_seconds = 300
        self.max_try_count = max_try_count
        self.try_delay = try_delay
        self.wait_for_seconds = wait_for_seconds

        self.pending_hosts = set(self.hosts)
        self.synced_hosts = set()
        self.bad_hosts = set()

        self.try_count = dict()
        self.start_time = None
        self.state_logged_time = None

    def run(self):
        """Block until replicas are synced or until timeout."""
        self.start_time = time.time()

        while self.pending_hosts:
            self.check_pending()

            if (time.time() - self.start_time) > self.wait_for_seconds:
                log.error(u'Не дождались синхронизации реплик "%s"', u'", "'.join(self.pending_hosts))
                raise BadReplicasError(self.pending_hosts)

            if not self.pending_hosts:
                break

            self.log_state()
            time.sleep(self.try_delay)

        log.info('Реплики синхронизировались за %.3f секунд', time.time() - self.start_time)

    def check_pending(self):
        for host in self.pending_hosts:
            self.check_host(host)

        self.pending_hosts -= self.synced_hosts
        self.pending_hosts -= self.bad_hosts

    def check_host(self, host):
        try:
            connection = self.conn_getter(host)
        except Exception:
            log.exception('Не можем подключиться к реплике "%s"', host)
            self.process_host_error(host)
        else:
            try:
                if self.is_synced(connection):
                    self.synced_hosts.add(host)
                    log.info('Реплика "%s" синхронизирована', host)
            except Exception:
                log.exception('Ошибка при выполнении запросов к реплике "%s"', host)
                self.process_host_error(host)
            finally:
                connection.close()

    def log_state(self):
        current_time = time.time()
        if self.state_logged_time is not None:
            if current_time - self.state_logged_time < self.report_interval_seconds:
                return

        for host in self.pending_hosts:
            log.info('Ждем синхронизации реплики "%s", время ожидания %.3f секунд',
                     host, time.time() - self.start_time)

        self.state_logged_time = current_time

    def process_host_error(self, host):
        try:
            self.try_count[host] += 1
        except KeyError:
            self.try_count[host] = 1

        if self.try_count[host] >= self.max_try_count:
            log.error('Реплика "%s" не работает - игнорируем', host)
            self.bad_hosts.add(host)
