# -*- coding: utf-8 -*-

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError

from common.models.geo import Station
from common.xgettext.i18n import gettext

from geosearch.views.point import (StationPointSearch, StopWordError,
                                   NoPointsError, TooShortError)


class StationForm(forms.Form):
    cityFrom = forms.CharField(max_length=200)
    cityFromId = forms.CharField(max_length=200, required=False)

    current = forms.BooleanField(required=False)
    other = forms.BooleanField(required=False)

    def __init__(self, *args, **kwargs):
        self.country_id = kwargs.pop('country_id', settings.OUR_COUNTRIES[0])

        kwargs['auto_id'] = "station_%s"
        super(StationForm, self).__init__(*args, **kwargs)

    def get_filter(self):
        if not self.is_valid():
            return None
        if self.cleaned_data['current'] and not self.cleaned_data['other']:
            return 'current'
        if not self.cleaned_data['current'] and self.cleaned_data['other']:
            return 'other'
        return None

    @classmethod
    def sample(cls, client_city):
        return {'cityFrom': Station.sample_title(client_city.region_id, client_city.country_id)}

    def clean(self):
        if 'cityFrom' not in self.cleaned_data:
            raise ValidationError(gettext(
                u"Введите название станции или города."
            ))

        try:
            points = StationPointSearch.find_point(
                self.cleaned_data['cityFrom'],
                self.cleaned_data['cityFromId'],
                country_id=self.country_id,
                country_filter_type=self.get_filter()
            )
        except StopWordError:
            raise ValidationError(gettext(
                u"Введенное вами название слишком общее.<br/>"
                u"Пожалуйста, уточните его."
            ))
        except TooShortError:
            raise ValidationError(gettext(
                u"Введенное вами название слишком короткое.<br/>"
                u"Пожалуйста, введите более длинное название (от 3 символов)."
            ))
        except NoPointsError:
            raise ValidationError(gettext(
                u"К сожалению, у нас нет информации о станции с таким "
                u"названием."
            ))

        self.cleaned_data['points'] = points

        return self.cleaned_data
