# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
import re
from itertools import groupby
from operator import itemgetter

from common.models.geo import Country, Point, Settlement, Station, StationMajority
from common.models.transport import TransportType
from common.utils.point_slugs import find_by_slug
from common.utils.text import similar_chars_to_cyr, similar_chars_to_lat, punto_variants, normalize
from geosearch.models import NameSearchIndex, DefaultPoint
from geosearch.views.pointlist import PointList

log = logging.getLogger(__name__)


PREFIXES = (
    r'город',
    r'пос[её]лок',
    r'станция',
    r'платформа',
    r'о\.п\.',
    r'оп',
    r'рзд',
    r'разъезд'
)

_prefixes_re = re.compile('^(?:(?:%s)\W*)*' % '|'.join(PREFIXES), re.U)


def strip_prefixes(title):
    return _prefixes_re.sub('', title)


_km_prefix_re = re.compile(r'(\d)км', re.U)


def fix_km_prefix(text):
    return _km_prefix_re.sub(r'\1 км', text)


def title_variants(title):
    if title is None:
        return []

    variants = [
        title,
        similar_chars_to_cyr(title),
        similar_chars_to_lat(title),
    ] + punto_variants(title)

    variants.extend(filter(strip_prefixes, variants))

    result = []
    seen = set()

    for v in variants:
        if v not in seen:
            result.append(v)
            seen.add(v)

    return result


def clean_title(title):
    return fix_km_prefix(normalize(title))


class PointSearchError(Exception):
    pass


class StopWordError(PointSearchError):
    WORDS = set([
        "пункт",
        "ост",
        "км",
        "на",
        "поворот",
        "трасса",
        "сан",
        "пасс",
        "им",
        "эль",
        "порт",
        "форт",
        "до",
        "де",
    ])

    def __init__(self, term):
        self.term = term

        super(StopWordError, self).__init__('term %r in stop words' % term)

    @classmethod
    def check(cls, term):
        term = term.strip(' -.()')
        return term in cls.WORDS


class TooShortError(PointSearchError):
    def __init__(self, term, min_length):
        super(TooShortError, self).__init__('term is too short (%d < %d)' % (len(term), min_length))


class PointNotFound(PointSearchError):
    def __init__(self, point_key):
        self.point_key = point_key
        super(PointNotFound, self).__init__("point with key %r not found", point_key)


class NoPointsError(PointSearchError):
    def __init__(self, term):
        self.term = term
        super(NoPointsError, self).__init__('no suitable points found for term %r' % term)


class InvalidPointKey(PointSearchError):
    def __init__(self, point_key):
        self.point_key = point_key
        super(InvalidPointKey, self).__init__('invalid point key %r', point_key)


class ExcessPointKey(PointSearchError):
    def __init__(self):
        super(ExcessPointKey, self).__init__('slug is already defined')


class InvalidSlug(PointSearchError):
    def __init__(self, slug):
        self.slug = slug
        super(InvalidSlug, self).__init__('invalid slug %r', slug)


class AllPointsFinder(object):
    @classmethod
    def find_all_points(cls, title):
        clean = clean_title(title)

        if StopWordError.check(clean):
            raise StopWordError(clean)

        if len(clean) < 2:
            raise TooShortError(clean, 2)

        for title_variant in title_variants(clean):
            points = cls.find_points(title_variant)

            if points:
                return points

        return []


class BasePointFinder(object):
    @classmethod
    def find_point(cls, title, t_type=None, point_key=None, slug=None, can_return_hidden=False):
        variants = []
        point = None

        if slug:
            point = find_by_slug(slug)
            if not point or (point.hidden and not can_return_hidden):
                raise InvalidSlug(slug)

        if point_key:
            if slug:
                raise ExcessPointKey()
            try:
                point = Point.get_by_key(point_key, use_hidden_manager=not can_return_hidden)
            except ValueError:
                raise InvalidPointKey(point_key)

        if title:
            all_points = cls.find_all_points(title)

            if t_type:
                variants = filter(cls.t_type_filter_factory(t_type), all_points)
            else:
                variants = all_points

        if point and point not in variants:
            variants.insert(0, point)

        if not variants:
            raise NoPointsError(title)

        return PointList(point, variants, title,
                         exact_variant=point is not None)

    @classmethod
    def t_type_filter_factory(cls, t_type):
        def function(point):
            if isinstance(point, Country):
                return True

            if t_type in point.type_choices_set:
                return True

            if t_type == 'suburban' and 'aeroex' in point.type_choices_set:
                return True

            if t_type == 'water' and point.type_choices_set.intersection(TransportType.WATER_TTYPE_CODES):
                return True

            if t_type == 'aeroex' and isinstance(point, Settlement):
                return True

            if isinstance(point, Station):
                water_t_types = TransportType.WATER_TTYPE_CODES

                if t_type in water_t_types and point.t_type.code in water_t_types:
                    return True

                return point.t_type.code == t_type

        return function


def find_filtered(match, model, title, function):
    result = NameSearchIndex.find(match, model, title)

    return filter(function, result.objects)


def not_hidden(o):
    return not o.hidden


class BaseSortKey(object):

    @classmethod
    def sort_key_factory(cls, title):
        def function(point):
            if DefaultPoint.is_default_title_point(title, point):
                return 0,

            if isinstance(point, Country):
                return 1,

            if isinstance(point, Settlement):
                return 2, point.majority_id, point.pk

            return 3, point.majority_id, point.title

        return function


class FirstBlockPoints(AllPointsFinder):
    """find_points возвращает первый непустой блок из _find_blocks"""

    @classmethod
    def find_points(cls, title):
        for block in cls._find_blocks(title):
            if block:
                return block

        return []


class SortedFirstBlockPoints(FirstBlockPoints):
    @classmethod
    def find_points(cls, title):
        first_block = super(SortedFirstBlockPoints, cls).find_points(title)

        first_block.sort(key=cls.sort_key_factory(title))

        return first_block


class BaseSuitableStation(object):
    @classmethod
    def suitable_station_factory(cls, settlements):
        def function(station):
            # Недостаточно важные
            if station.majority_id >= StationMajority.NOT_IN_SEARCH_ID:
                return False

            # Скрытые
            if station.hidden:
                return False

            # не предлагать для уточнения на выбор - город и одноименную станцию внутри города
            # (RASP-1874)

            # если города нет, то пропускаем
            if not station.settlement_id:
                return True

            station_settlement = None

            for s in settlements:
                if s.id == station.settlement_id:
                    station_settlement = s
                    break

            if station_settlement:
                if normalize(station_settlement.title) == normalize(station.title):
                    return False

            return True

        return function


class BasePointSearch(AllPointsFinder):
    """find_points для основной формы и поиска по станция"""

    # Вернуть город, если он один и нет неавтобусных станций, не принадлежащих данному городу
    RETURN_SINGLE_CITY_IF_ONE_CITY_AND_NO_OUTSIDE_NON_BUS_STATIONS = True

    @classmethod
    def find_points(cls, title):
        blocks = cls._find_blocks(title)
        return cls._sorted_points(blocks, title)

    @classmethod
    def _sorted_points(cls, blocks, title):
        sorted_blocks = sorted(blocks, key=itemgetter(0))

        sort_key = cls.sort_key_factory(title)

        seen = set()
        points = []

        for _priority, blocks in groupby(sorted_blocks, key=itemgetter(0)):
            priority_points = set(
                point
                for _priority, block_points in blocks
                for point in block_points
                if point not in seen
            )

            seen.update(priority_points)
            points.extend(sorted(priority_points, key=sort_key))

        return points

    @classmethod
    def _find_blocks(cls, title):
        """
        Возвращает списки найденных объектов с важностью каждого списка
        """
        settlements = find_filtered('exact', Settlement, title, not_hidden)

        yield 0, settlements

        suitable_station = cls.suitable_station_factory(settlements)

        stations = find_filtered('exact', Station, title, suitable_station)

        if cls.RETURN_SINGLE_CITY_IF_ONE_CITY_AND_NO_OUTSIDE_NON_BUS_STATIONS:
            if len(settlements) == 1:
                settlement = settlements[0]

                outside_non_bus_stations = [
                    s
                    for s in stations
                    if s.settlement_id != settlement.id and s.t_type.code != 'bus'
                ]

                if not outside_non_bus_stations:
                    return

        yield 0, stations

        yield 0, find_filtered('exact', Country, title, not_hidden)

        # words
        settlements = find_filtered('words', Settlement, title, not_hidden)
        yield 1, settlements

        suitable_station = cls.suitable_station_factory(settlements)
        yield 1, find_filtered('words', Station, title, suitable_station)

        yield 1, find_filtered('words', Country, title, not_hidden)

        # codes
        code = title

        yield 1, Settlement.hidden_manager.get_list(iata__iexact=code)

        yield 1, Station.code_manager.get_list_by_code(code)


class PointSearch(BasePointFinder, BasePointSearch, BaseSuitableStation, BaseSortKey):
    pass


class BaseSuggestsPointSearch(BasePointSearch):
    RETURN_SINGLE_CITY_IF_ONE_CITY_AND_NO_OUTSIDE_NON_BUS_STATIONS = False


class SuggestsPointSearch(BasePointFinder, BaseSuggestsPointSearch, BaseSuitableStation, BaseSortKey):
    pass


class StationPointSearch(BasePointSearch, BaseSortKey):
    @classmethod
    def suitable_station_factory(cls, settlements):
        def function(station):
            if station.hidden:
                return False

            if station.majority_id > StationMajority.NOT_IN_SEARCH_ID:
                return False

            # RASP-9057
            if station.majority_id == StationMajority.NOT_IN_SEARCH_ID and not station.is_mta:
                return False

            return True

        return function

    @classmethod
    def find_point(cls, title, point_key=None, country_id=None,
                   country_filter_type=None):
        if point_key:
            # Точное указание
            try:
                point = Point.get_by_key(point_key)

                if not isinstance(point, (Settlement, Station)):
                    raise ValueError("Wrong point type: %s (%s)", type(point),
                                     point_key)

            except (ValueError, Station.DoesNotExist, Settlement.DoesNotExist):
                log.warning("Invalid point key %r", point_key, exc_info=True)

            else:
                return PointList(point)

        # Поиск по названию
        variants = cls.find_all_points(title)

        for point in variants:
            if point.type == 'country':
                return PointList(point, term=title)

        if country_id is not None and country_filter_type is not None:
            variants = filter(
                cls.country_filter_factory(country_id, country_filter_type),
                variants
            )

        if not variants:
            raise NoPointsError(title)

        if len(variants) == 1:
            return PointList(variants[0], term=title)

        exact_variants = [
            p
            for p in variants
            if p.title.upper() == title.upper()
        ]

        if len(exact_variants) == 1:
            # Если в списке точно совпадает только одна точка, возвращаем её
            return PointList(exact_variants[0], term=title)

        return PointList(None, variants, term=title)

    @classmethod
    def country_filter_factory(cls, country_id, country_filter_type):
        if country_filter_type == 'current':
            return (lambda point: point.country_id == country_id)
        elif country_filter_type == 'other':
            return (lambda point: point.country_id != country_id)
        else:
            raise ValueError(
                "Invalid country filter type: %r" % country_filter_type
            )


class MobilePointSearch(BasePointFinder, SortedFirstBlockPoints, BaseSuitableStation, BaseSortKey):
    @classmethod
    def _find_blocks(cls, title):
        settlements = find_filtered('exact', Settlement, title, not_hidden)

        suitable_station = cls.suitable_station_factory(settlements)

        stations = find_filtered('exact', Station, title, suitable_station)

        if len(settlements) == 1:
            settlement = settlements[0]

            stations = [
                s
                for s in stations
                if s.settlement_id != settlement.id and s.t_type.code != 'bus'
            ]

        yield settlements + stations

        settlements = find_filtered('words', Settlement, title, not_hidden)
        stations = find_filtered('words', Station, title, suitable_station)

        yield settlements + stations

        # codes
        code = title

        yield Settlement.hidden_manager.get_list(iata__iexact=code)

        yield Station.code_manager.get_list_by_code(code)


class SuburbanPointSearch(BasePointFinder, SortedFirstBlockPoints):
    """Поиск для электричек - сначала станции, потом города"""

    @classmethod
    def _find_blocks(cls, title):
        for match in ['exact', 'words']:
            stations = find_filtered(match, Station, title, cls.suitable_station)
            settlements = find_filtered(match, Settlement, title,
                                        cls.suitable_settlement_factory(stations))

            yield stations + settlements

        yield filter(cls.suitable_station, Station.code_manager.get_list_by_code(title))

    @classmethod
    def suitable_station(cls, station):
        if station.t_type.code not in ['train', 'plane']:
            return False

        # Недостаточно важные
        if station.majority_id >= StationMajority.NOT_IN_SEARCH_ID:
            return False

        # Скрытые
        if station.hidden:
            return False

        if station.t_type.code == 'plane':
            if not station.has_aeroexpress:
                return False

            # RASP-12633 В случае если название станции совпадает с названием города, брать станцию
            # (правило Коломны) - кроме тех случаев, когда эта станция имеет вид транспорта
            # "Самолет"
            if station.L_title() == station.settlement.L_title():
                return False

        return True

    @classmethod
    def suitable_settlement_factory(cls, stations):
        def function(settlement):
            if settlement.hidden:
                return False

            norm_title = normalize(settlement.title)

            # Если уже найдена одноименная станция города, то пропускаем этот город
            for s in stations:
                if s.settlement_id == settlement.id and normalize(s.title) == norm_title:
                    return False

            return True

        return function

    @classmethod
    def sort_key_factory(cls, title):
        def function(point):
            # сначала Домодедово и Внуково, потом города, потом остальные станции (RASP-2248,
            # RASP-3375)
            if isinstance(point, Settlement):
                return 3, point.majority_id, point.pk
            if point.id in (9600215, 9600216):
                return 1, point.majority_id
            else:
                return 2, point.majority_id, point.title

        return function


class HiddenCitySearch(BasePointFinder, FirstBlockPoints, BaseSuitableStation):
    """Поиск скрытых городов - показываем ближайшие"""

    @classmethod
    def _find_blocks(cls, title):
        for match in ['exact', 'words']:
            countries = NameSearchIndex.find(match, Country, title).objects
            settlements = NameSearchIndex.find(match, Settlement, title).objects

            suitable_station = cls.suitable_station_factory(settlements)
            stations = find_filtered(match, Station, title, suitable_station)

            yield countries + settlements + stations

        countries = NameSearchIndex.find('words', Country, title).objects
        settlements = NameSearchIndex.find('words', Settlement, title).objects

        suitable_station = cls.suitable_station_factory(settlements)
        stations = find_filtered('words', Station, title, suitable_station)

        yield countries + settlements + stations

        # Коды городов
        code = title

        yield list(Settlement.hidden_manager.filter(iata=code))

        # Коды станций
        yield Station.code_manager.get_list_by_code(code)
