# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import functools
import locale

from django.conf import settings
from django.utils import translation
from django.utils.encoding import smart_str

from common.models.geo import Country, Point, Settlement, Station, StationMajority
from common.models_utils import fetch_related
from common.utils.text import normalize

from geosearch.views.cache import have_common_direction


_inverted_key = functools.cmp_to_key(lambda a, b: cmp(b, a))


def _locale_inverted_key(string):
    return _inverted_key(locale.strxfrm(smart_str(string)))


class FakeSuggest(object):
    def __init__(self, point, **kwargs):
        self.point = point

        for key, value in kwargs.items():
            setattr(self, key, value)

    def has_common_direction(self, other_point):
        """Определяет, есть ли общее внешнее направление у текущего саггеста и второго пункта"""
        if not other_point or not isinstance(other_point, Station):
            return False

        if self.type != 'station':
            return False

        return have_common_direction(self.object_id, other_point.id)

    def get_city_priority(self, our_countries):
        if self.type == 'settlement':
            if self.majority == 1:  # Столица наших стран
                if self.country_id in our_countries:
                    return 4

            elif self.majority == 3:  # Город миллионник
                return 3

            elif self.majority == 2:  # Областной центр наших стран
                if self.country_id in our_countries:
                    return 2

        return 1

    @classmethod
    def from_point(cls, point):
        if isinstance(point, Country):
            return cls.from_country(point)
        if isinstance(point, Settlement):
            return cls.from_settlement(point)
        if isinstance(point, Station):
            return cls.from_station(point)
        raise NotImplementedError("Неизвестный класс точки")

    @classmethod
    def from_country(cls, country):
        lang = translation.get_language()
        title = country.L_title(lang=lang)
        return cls(country,
                   title=title,
                   full_title=title,
                   ttype=None,
                   type='country',
                   object_id=country.pk,
                   majority=1,
                   region_id=None,
                   country_id=country.id,
                   **{'full_title_' + lang: title,
                      'title_' + lang: title})

    @classmethod
    def from_settlement(cls, settlement):
        lang = translation.get_language()
        title = settlement.L_title(lang=lang)
        return cls(settlement,
                   title=title,
                   full_title=title,
                   ttype=None,
                   type='settlement',
                   object_id=settlement.pk,
                   majority=settlement.majority_id,
                   region_id=settlement.region_id,
                   country_id=settlement.country_id,
                   **{'full_title_' + lang: title,
                      'title_' + lang: title})

    @classmethod
    def from_station(cls, station):
        lang = translation.get_language()
        title = station.L_title(lang=lang)
        return cls(station,
                   title=title,
                   full_title=title,
                   ttype=station.t_type.code,
                   type='station',
                   object_id=station.pk,
                   majority=station.majority_id,
                   region_id=station.region_id,
                   country_id=station.country_id,
                   **{'full_title_' + lang: title,
                      'title_' + lang: title})

    @classmethod
    def sort_key_factory(cls, query=None, client_city=None, other_point=None,
                         ttype=None, national_version=None):
        if client_city:
            client_region_id = getattr(client_city, 'region_id', None)
            client_country_id = getattr(client_city, 'country_id', None)
        else:
            client_region_id = client_country_id = None

        norm_query = normalize(query)

        # оптимизация вызовов suggest.L_title()
        title_attr = 'title_%s' % translation.get_language()

        def L_title(suggest):
            return getattr(suggest, title_attr) or suggest.title

        factory = (
            cls.plane_sort_key_factory
            if ttype == 'plane' else
            cls.common_sort_key_factory
        )

        return factory(L_title, norm_query, client_region_id,
                       client_country_id, other_point, national_version)

    @classmethod
    def common_sort_key_factory(cls, L_title, norm_query, client_region_id,
                                client_country_id, other_point,
                                national_version):
        our_countries = settings.LOCAL_KUBR.get(national_version,
                                                settings.LOCAL_KUBR['ru'])

        # RASP-6114
        def function(suggest):
            title = L_title(suggest)

            if not suggest.object_id:
                return -10, _locale_inverted_key(title)

            if suggest.type == 'country':
                return -9, _locale_inverted_key(title)

            norm_title = normalize(title) if norm_query else None

            if suggest.type == 'settlement':
                suggest_type_priority = 3
            elif suggest.type == 'station' and suggest.ttype == 'plane':
                suggest_type_priority = 2
            else:
                suggest_type_priority = 1

            # FIXME: Вернуть has_common_direction
            return (
                norm_title is not None and norm_title.startswith(norm_query),
                suggest.get_city_priority(our_countries),
                bool(client_region_id
                     and suggest.region_id == client_region_id),
                bool(client_country_id
                     and suggest.country_id == client_country_id),
                norm_title is not None and norm_title == norm_query,
                suggest_type_priority,
                -suggest.majority,
                _locale_inverted_key(title)
            )

        return function

    @classmethod
    def plane_sort_key_factory(cls, L_title, norm_query, client_region_id,
                               client_country_id, other_point,
                               national_version):
        our_countries = settings.LOCAL_KUBR.get(national_version,
                                                settings.LOCAL_KUBR['ru'])

        # RASP-6635
        def function(suggest):
            title = L_title(suggest)

            if not suggest.object_id:
                return -10, _locale_inverted_key(title)

            if suggest.type == 'country':
                return -9, _locale_inverted_key(title)

            norm_title = normalize(title) if norm_query else None

            return (
                norm_title is not None and norm_title.startswith(norm_query),
                suggest.get_city_priority(our_countries),
                bool(client_region_id
                     and suggest.region_id == client_region_id),
                bool(client_country_id
                     and suggest.country_id == client_country_id),
                -(suggest.majority or StationMajority.MAX_ID),
                suggest.type == 'settlement',
                _locale_inverted_key(title),
            )

        return function


def rearrange_variants(variants, client_city=None, query_phrase=None,
                       national_version=None):
    """Сортировка уточнений по правилам саггестов"""

    suggests = [FakeSuggest.from_point(point) for point in variants]

    sort_key = FakeSuggest.sort_key_factory(query_phrase, client_city,
                                            national_version=national_version)

    suggests.sort(key=sort_key, reverse=True)

    return [s.point for s in suggests]


def fetch_related_for_points(points):
    stations = [point for point in points if isinstance(point, Station)]
    fetch_related(stations, 'settlement', 'country', 'region', 'district', model=Station)

    settlements = [point for point in points if isinstance(point, Settlement)]
    fetch_related(settlements, 'country', 'district', 'region', model=Settlement)


class PointList(object):
    def __init__(self, point, variants=None, term=None, final_variant=False, exact_variant=False):
        self.point = point
        self.variants = variants or ()
        self.term = term
        # Признак, что в point лежит финальный вариант, и уточения уже не нужны
        self.final_variant = final_variant
        self.exact_variant = exact_variant  # Признак, что передали идентификатор точки
        self.dont_rearrange = False
        self.allow_reduce = True

        fetch_related_for_points(list(self.variants) + ([point] if point else []))

    def has_variants(self):
        return len(self.variants) > 1 and not (self.final_variant or self.exact_variant)

    def mobile_has_variants(self):
        return len(self.variants) > 1 and not self.exact_variant

    @property
    def variants_without_countries(self):
        return (point for point in self.variants if not isinstance(point, Country))

    def __unicode__(self):
        def pr(point):
            if point:
                return "%s (%s/%s)" % (point.title, point.pk, hasattr(point, 'majority_id') and point.majority_id or 1)
            return ""
        return pr(self.point) \
                + " [%s]" % ", ".join((pr(point) for point in self.variants)) \
                + "(final: %s, exact: %s)" % (self.final_variant, self.exact_variant)

    def __str__(self):
        return self.__unicode__().encode('utf-8')

    def repr(self):
        def pr(point):
            if point:
                return "%s (%s/%s)" % (point.title, point.pk, hasattr(point, 'majority_id') and point.majority_id or 1)
            return ""
        return {
            'point': pr(self.point),
            'points': [pr(point) for point in self.variants],
            'final_variant': self.final_variant,
            'exact_variant': self.exact_variant
        }

    @classmethod
    def restore(cls, dump):
        point = Point.get_by_key(dump['point'])
        points = [Point.get_by_key(point_key) for point_key in dump['points']]
        return cls(point, points, final_variant=bool(dump.get('final_variant')), exact_variant=bool(dump.get('exact_variant')))

    def dump(self):
        return {
            'point': self.point.point_key,
            'points': [point.point_key for point in self.variants],
            'final_variant': self.final_variant,
            'exact_variant': self.exact_variant
        }

    def rearrange_variants(self, client_city, query_phrase=None,
                           national_version=None):
        """Сортировка уточнений по правилам саггестов"""
        self.variants = rearrange_variants(self.variants, client_city, query_phrase)

    def process(self):
        """ Пост-обработка списка вариантов. Проверка, надо ли выводить города или области у станций с одинаковыми названиями. """
        if len(self.variants) <= 1:
            return

        def stations_only(points):
            return (p for p in points if isinstance(p, Station))

        for v in stations_only(self.variants):
            for w in stations_only(self.variants):
                if v != w and v.title.lower() == w.title.lower() and isinstance(v, Station) and isinstance(w, Station):
                    if ((v.settlement_id is not None and v.settlement_id == w.settlement_id)
                            or (v.region_id is not None and v.region_id == w.region_id)):
                        v.show_direction = True

                    elif v.settlement_id is not None and w.settlement_id is not None and v.settlement.title == w.settlement.title:
                        v.force_show_region = True

        # Не показываем у станции ссылку на город, если этот город есть в списке
        for v in stations_only(self.variants):
            if v.settlement in self.variants:
                v.hide_settlement = True

    def __eq__(self, l2):
        if self.point == l2.point and self.variants == l2.variants:
            return True
        return False

    def __ne__(self, l2):
        return not self.__eq__(l2)
