# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from common.models.geo import ReplaceException, Settlement, Station, ExternalDirectionMarker
from common.models.transport import TransportType
from common.models_utils import fetch_related
from common.utils.iterrecipes import unique_everseen

from geosearch.views.cache import ExternalDirectionCache, SuburbanZoneCache, StationCache
from geosearch.views.pointlist import PointList


log = logging.getLogger(__name__)


# Замена города на станцию. RASP-2338

# Город, направление, id станции замены
POINT_REPLACEMENTS = [
    ['москва', 'msk_bel', 2000006],
    ['москва', 'msk_gor', 2000001],
    ['москва', 'msk_kaz', 2000003],
    ['москва', 'msk_kiv', 2000007],
    ['москва', 'msk_kur', 2000001],
    ['москва', 'msk_len', 2006004],
    ['москва', 'msk_pav', 2000005],
    ['москва', 'msk_sav', 2000009],
    ['москва', 'msk_yar', 2000002],
    ['москва', 'msk_riz', 2000008],
    ['москва', 'msk_mkzd', 9601334],
    ['санкт-петербург', 'spb_vit', 9602496],
    ['санкт-петербург', 'spb_vyb', 9602497],
    ['санкт-петербург', 'spb_msk', 9602494],
    ['санкт-петербург', 'spb_pri', 9602497],
    ['санкт-петербург', 'spb_lad', 9602497],
    ['санкт-петербург', 'spb_bal', 9602498],
]

# Эти города не надо заменять на станции
PRESERVED_CITIES_IDS = {
    213,  # Москва
    2,  # Санкт-Петербург
    11,  # Рязань
    15,  # Тула
    147,  # Харьков
    157  # Минск
}


class SamePointError(Exception):
    def __init__(self, point_list_from, point_list_to):
        self.point_list_from = point_list_from
        self.point_list_to = point_list_to

        super(SamePointError, self).__init__("Couldn't discriminate from and to points lists")


def process_points_lists(point_list_from, point_list_to, client_city=None, suburban=False,
                         disable_reduce_from=False, disable_reduce_to=False, disable_replace=False):
    """Обработка полученных списков городов, запрошенных юзером в поисковойформе"""

    log.debug('Вход: %s, %s' % (point_list_from, point_list_to))

    # RASP-3259: Выбирать из множества одноименных станций те, что в регионе пользователя
    if client_city and point_list_from.variants and point_list_to.variants:
        user_region_id = client_city.region_id

        if user_region_id:
            if not point_list_from.point:
                for v in point_list_from.variants:
                    if getattr(v, 'region_id', None) == user_region_id:
                        point_list_from.point = v
                        break

            if not point_list_to.point:
                for v in point_list_to.variants:
                    if getattr(v, 'region_id', None) == user_region_id:
                        point_list_to.point = v
                        break

    # RASP-4171, RASP-6631: Выбираем станции одного внешнего направления
    if not point_list_from.point or not point_list_to.point:
        if point_list_from.variants and point_list_to.variants:
            stations_from = [v for v in point_list_from.variants if isinstance(v, Station)]
            stations_to = [v for v in point_list_to.variants if isinstance(v, Station)]

            markers_from = ExternalDirectionCache.markers_by_stations(stations_from)
            markers_to = ExternalDirectionCache.markers_by_stations(stations_to)

            common = set()

            for marker_from in markers_from:
                for marker_to in markers_to:
                    if marker_from.external_direction_id == marker_to.external_direction_id:
                        common.add((marker_from.station, marker_to.station))

            if len(common) == 1:  # Только одна пара совпадающих по направлению станций
                station_from, station_to = common.pop()

                if point_list_from.has_variants() and not disable_reduce_from:
                    point_list_from.point = station_from
                    point_list_from.final_variant = True

                if point_list_to.has_variants() and not disable_reduce_to:
                    point_list_to.point = station_to
                    point_list_to.final_variant = True

    # RASP-3435: Выбираем станции в одной пригородной зоне
    if not point_list_from.point or not point_list_to.point:
        if point_list_from.has_variants() and point_list_to.has_variants():
            zones_from = [v.suburban_zone_id for v in point_list_from.variants]
            zones_to = [v.suburban_zone_id for v in point_list_to.variants]

            common_zone = None

            for zone_from in zones_from:
                for zone_to in zones_to:
                    if zone_from == zone_to:
                        common_zone = zone_from
                        break

            if common_zone:
                for point_list in [point_list_from, point_list_to]:
                    if not point_list.point:
                        for v in point_list.variants:
                            if v.suburban_zone_id == common_zone:
                                point_list.point = v
                                break

    # RASP-10218 Выбрать города на одном внешнем направлении
    if suburban:
        if not point_list_from.point or not point_list_to.point:
            if point_list_from.variants and point_list_to.variants:
                common = set()

                point_directions_from = ExternalDirectionCache \
                    .ids_by_points_in_bulk(point_list_from.variants)

                point_directions_to = ExternalDirectionCache \
                    .ids_by_points_in_bulk(point_list_to.variants)

                for point_from, directions_from_ids in point_directions_from:
                    for point_to, directions_to_ids in point_directions_to:
                        if not directions_from_ids.isdisjoint(directions_to_ids):
                            common.add((point_from, point_to))

                if len(common) == 1:  # Только одна пара совпадающих по направлению пунктов
                    point_from, point_to = common.pop()

                    if point_list_from.has_variants() and not disable_reduce_from:
                        point_list_from.point = point_from
                        point_list_from.final_variant = True

                    if point_list_to.has_variants() and not disable_reduce_to:
                        point_list_to.point = point_to
                        point_list_to.final_variant = True

    # Если города до сих пор не выбраны, то берем первый вариант
    # Подразумевается, что вариантов больше одного
    if not point_list_from.point:
        point_list_from.point = point_list_from.variants[0]

        if len(point_list_from.variants) > 1:
            point_list_from.allow_reduce = False

    if not point_list_to.point:
        point_list_to.point = point_list_to.variants[0]

        if len(point_list_to.variants) > 1:
            point_list_to.allow_reduce = False

    # Выбираем первый вариант, не равный другому выбранному
    if point_list_from.point == point_list_to.point:
        if point_list_to.has_variants():
            for point in point_list_to.variants:
                if point != point_list_from.point:
                    point_list_to.point = point

        elif point_list_from.has_variants():
            for point in point_list_from.variants:
                if point != point_list_to.point:
                    point_list_from.point = point

        else:
            raise SamePointError(point_list_from, point_list_to)

    # Убираем из вариантов другого конца точно указанные пункты
    if point_list_to.exact_variant:
        point_list_from.variants = [variant for variant in point_list_from.variants
                                    if variant != point_list_to.point]

    if point_list_from.exact_variant:
        point_list_to.variants = [variant for variant in point_list_to.variants
                                  if variant != point_list_from.point]

    # Колдунство замены города на станцию
    if not disable_replace and (
        allow_replace(point_list_from.point, point_list_to.point, suburban) or
        allow_replace(point_list_to.point, point_list_from.point, suburban)
    ):

        log.debug('Колдунство замены города на станцию')
        log.debug('До замены: %s, %s' % (point_list_from, point_list_to))

        # RASP-11756
        if not disable_reduce_from:
            point_list_from = replace_point(point_list_from, point_list_to)

        # RASP-11756
        if not disable_reduce_to:
            point_list_to = replace_point(point_list_to, point_list_from)

        log.debug('После замены: %s, %s' % (point_list_from, point_list_to))

    # RASP-9636 при поиске автобуса внутри города требовать уточнения
    def process_is_busstop_in_settlement(station_pointlist, settlement_pointlist):
        station = station_pointlist.point
        settlement = settlement_pointlist.point

        if not station.is_station:
            return False

        if station.is_station and station.settlement == settlement and station.t_type_id == TransportType.BUS_ID:
            # Заполнить всеми автобусными станциями этого города
            variants = list(settlement.station_set.filter(t_type_id=TransportType.BUS_ID, hidden=False)
                            .exclude(id=station.id).order_by('majority__id', 'title'))

            if variants:
                log.debug('До смены города на список станций города: %s' % settlement_pointlist)
                settlement_pointlist.point = variants[0]

                # Если больше одной станции с важностью выше остановочного пункта
                # тогда выводим список вариантов
                if len(variants) > 1 and variants[1].majority_id < 4:
                    settlement_pointlist.variants = variants
                    settlement_pointlist.exact_variant = False
                    settlement_pointlist.dont_rearrange = True
                log.debug('После смены города на список станций города: %s' % settlement_pointlist)
                # иначе сразу ищем до неё
                return True

    if not point_list_from.has_variants() and not point_list_to.has_variants():
        process_is_busstop_in_settlement(point_list_from, point_list_to)
        process_is_busstop_in_settlement(point_list_to, point_list_from)

    # RASP-4084: сужение города до станции
    log.debug('До сужения города до станции: %s, %s' % (point_list_from, point_list_to))

    # RASP-11756: отключаем сужение
    if not disable_reduce_from:
        point_list_from = reduce_city(point_list_to.point, point_list_from, suburban=suburban)

    # RASP-11756: отключаем сужение
    if not disable_reduce_to:
        point_list_to = reduce_city(point_list_from.point, point_list_to, suburban=suburban)

    log.debug('После сужения города до станции: %s, %s' % (point_list_from, point_list_to))

    if point_list_from.point == point_list_to.point:
        raise SamePointError(point_list_from, point_list_to)

    log.debug('Выход: %s, %s' % (point_list_from, point_list_to))

    return point_list_from, point_list_to


def allow_replace(point_from, point_to, suburban):
    """Определяет, можно ли применять колдунство по замене города на станцию"""
    if ReplaceException.check_points(point_from, point_to):

        return False

    if ReplaceException.check_points(point_to, point_from):
        return False

    if suburban:
        return allow_replace_for_main_cities(point_from, point_to)  # RASP-4467

    # RASP-5788
    if isinstance(point_from, Settlement) and isinstance(point_to, Station):
        if point_from.id == point_to.settlement_id:
            return True

    return False


def allow_replace_for_main_cities(point1, point2):
    """
    Запрещать колдунство, если обе станции принадлежат разным главным городам направлений. RASP-4467
    """

    cities = []

    for point in [point1, point2]:
        if isinstance(point1, Settlement):
            city = point1
        elif isinstance(point1, Station):
            if point1.settlement:
                city = point1.settlement
            else:
                return True
        else:
            return False

        cities.append(city)

    zone1 = SuburbanZoneCache.by_settlement_id(cities[0].pk)
    zone2 = SuburbanZoneCache.by_settlement_id(cities[1].pk)

    if zone1 and zone2 and zone1 != zone2:
        return False

    return True


def replace_point(point_list1, point_list2):
    """Колдунство. Замена города на станцию. RASP-2338"""

    if isinstance(point_list2.point, Station):
        stations = [point_list2.point]
    elif isinstance(point_list2.point, Settlement):
        stations = StationCache.by_settlement(point_list2.point)
    else:
        return point_list1

    dirs = dict((edm.external_direction.code, True)
                for edm in ExternalDirectionCache.markers_by_stations(stations))

    for r in POINT_REPLACEMENTS:
        if point_list1.point.title.lower() == r[0] and r[1] in dirs:
            station = Station.objects.get(pk=r[2])

            return PointList(station, [station], point_list1.term)

    return point_list1


def reduce_city(station, point_list, suburban):
    """RASP-3375, RASP-4084 - сужение города до станции"""

    if point_list.final_variant:
        return point_list

    if not need_to_reduce(station, point_list, suburban):
        return point_list

    city = point_list.point

    city_stations = StationCache.by_settlement(city)

    def get_variants(station):
        """ Уточнения при сужении """

        stations = sorted((
            s for s in city_stations
            if s.t_type_id in [TransportType.TRAIN_ID, TransportType.PLANE_ID] and s.majority_id <= 2
        ), key=lambda s: s.majority_id)

        variants = [station] + stations

        for v in point_list.variants:
            if v != city:
                variants.append(v)

        return list(unique_everseen(variants))

    if isinstance(station, Station):
        directions = ExternalDirectionCache.by_station(station)
    elif isinstance(station, Settlement):
        directions = ExternalDirectionCache.by_settlement(station)
    else:
        directions = []

    if len(directions):
        direction = directions[0]

        if direction.base_station and direction.base_station.settlement == city:
            return PointList(direction.base_station, get_variants(direction.base_station),
                             point_list.term, final_variant=True)

    stations = [
        s for s in city_stations
        if s.t_type_id == TransportType.TRAIN_ID and s.majority_id <= 4 and s.title.lower() == city.title.lower()
    ]

    if len(stations):
        station = stations[0]

    else:
        stations = sorted((
            s for s in city_stations
            if s.t_type_id in [TransportType.TRAIN_ID, TransportType.PLANE_ID]
        ), key=lambda s: s.majority_id)

        if len(stations):
            # Ищем станцию того же внешнего направления, что и station
            city_markers = ExternalDirectionCache.markers_by_stations(stations)
            fetch_related(city_markers, 'station', model=ExternalDirectionMarker)
            city_markers = sorted(city_markers, key=lambda m: m.station.majority_id)

            stations_directions = set(d.id for d in directions)

            for marker in city_markers:
                if marker.external_direction_id in stations_directions:
                    return PointList(marker.station, get_variants(marker.station), point_list.term,
                                     final_variant=True)

            station = stations[0]
        else:
            return point_list

    return PointList(station, get_variants(station), point_list.term, final_variant=True)


def need_to_reduce(station, point_list, suburban):
    if not point_list.allow_reduce:
        return False

    city = point_list.point
    if not isinstance(city, Settlement):
        return False

    # Стоп-лист
    if city.id in PRESERVED_CITIES_IDS:
        return False

    if suburban:
        return True

    if not isinstance(station, Station):
        return False

    # Аэропорты не сужаем
    if station.t_type_id == TransportType.PLANE_ID:
        return False

    return station.settlement_id == city.id


def iter_extended_points(point_from, point_to, suburban):
    city_from = isinstance(point_from, Station) and point_from.settlement_id and point_from.settlement or point_from
    city_to = isinstance(point_to, Station) and point_to.settlement_id and point_to.settlement or point_to

    if city_from != city_to:
        point_list_from = PointList(city_from, [city_from])
        point_list_to = PointList(city_to, [city_to])

        point_list_from, point_list_to = process_points_lists(
            point_list_from, point_list_to, suburban=suburban
        )
        city_from, city_to = point_list_from.point, point_list_to.point

        if city_from != point_from:
            yield city_from, point_to

        if city_to != point_to:
            yield point_from, city_to

        if city_from != point_from and city_to != point_to:
            yield city_from, city_to
