# -*- coding: utf-8 -*-

import cPickle as pickle
import logging
import os

from django.conf import settings

from common.db.switcher import switcher
from common.models.geo import Station
from common.settings.utils import define_setting


define_setting('GEOMETRY_BASE_PATH', default='/resources/geometry/geometry')
define_setting('PRELOAD_MAPPING_GEOMETRY', default=False)


log = logging.getLogger(__name__)


class GeometryStorage(object):
    def __init__(self, filename):
        self.filename = filename
        self.data = None

    def get_data(self):
        if self.data is None:
            self.preload()

        return self.data

    def preload(self):
        self.data = None

        if settings.PRELOAD_MAPPING_GEOMETRY:
            if not getattr(settings, 'GEOMETRY_BASE_PATH', None):
                log.error(u'Не заполнен GEOMETRY_BASE_PATH')
                return

            data_path = os.path.join(settings.GEOMETRY_BASE_PATH, switcher.get_db_alias(), self.filename)

            try:
                with open(data_path, 'rb') as f:
                    data = pickle.load(f)
            except:
                log.exception(u'Ошибка при загрузке файла данных %s' % data_path)
                return

            self.data = data


class TrainPathDrawer(object):
    storage = GeometryStorage('railways.pickle')

    def __init__(self, thread):
        self.thread = thread

        data = self.storage.get_data()

        self.segments = data['segments'] if data else {}
        self.stations = data['stations'] if data else {}

    def get_station_coords(self, station_id):
        return self.stations.get(station_id)

    def get_arc(self, st1_id, st2_id):
        if not self.segments:
            return None

        segment = (self.segments.get((st1_id, st2_id, self.thread.number)) or
                   self.segments.get((st1_id, st2_id, None)))

        return segment


class StationCoordsStorage(object):
    def __init__(self):
        self.data = None

    def preload(self):
        coords = Station.objects.\
            filter(longitude__isnull=False, latitude__isnull=False).\
            values_list('id', 'longitude', 'latitude')

        self.data = {station_id: [lng, lat] for station_id, lng, lat in coords}

    def get(self, station_id):
        if self.data is None:
            self.preload()

        return self.data.get(station_id)


StationCoordsStorage = StationCoordsStorage()


class LimePathDrawer(object):
    storage = GeometryStorage('limepaths.pickle')

    def __init__(self):
        data = self.storage.get_data()

        self.segments = data['segments'] if data else {}

    def get_station_coords(self, station_id):
        return StationCoordsStorage.get(station_id)

    def get_arc(self, st1_id, st2_id):
        if not self.segments:
            return None

        # Восстановление порядка станций согласно RoutePath._order_stations
        if st1_id < st2_id:
            key = (st1_id, st2_id)
            direct = True
        else:
            key = (st2_id, st1_id)
            direct = False

        try:
            direct_segment, back_segment, for_two_directions = self.segments[key]
        except KeyError:
            return None

        if direct_segment and (direct or for_two_directions):
            return direct_segment
        elif back_segment and (not direct or for_two_directions):
            return back_segment

        return None
