# -*- coding: utf-8 -*-

import json
from itertools import izip

import numpy as np
from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.models.schedule import Company
from common.models.transport import TransportType
from common.utils.fields import TrimmedCharField
from mapping.utils import interpolate_coords


class Flight(models.Model):
    station_from = models.ForeignKey('www.Station', related_name='departed_flights')
    station_to = models.ForeignKey('www.Station', related_name='arriving_flights')
    departure = models.IntegerField()
    arrival = models.IntegerField()

    lng = models.FloatField()
    lat = models.FloatField()
    zoom = models.IntegerField()

    number = models.CharField(max_length=100)
    route_uid = models.CharField(max_length=100, null=True)
    title = models.CharField(max_length=100)
    thread_uid = models.CharField(max_length=100, null=True)
    company_title = models.CharField(max_length=100, null=True)

    t_code = models.CharField(max_length=10)

    def is_valid(self):
        return self.station_from and self.station_to and \
            self.station_from.longitude and self.station_to.longitude

    def in_air(self, now, threshold):
        return self.departure - threshold < now and self.arrival + threshold > now

    def fill_fields(self):
        # тут мы рассчитываем на то, что Company - закэшировано
        try:
            self.company_title = Company.hidden_manager.get(id=self.company_id)
        except Company.DoesNotExist:
            self.company_title = None

        self.t_code = 'plane'

        if self.thread:
            self.thread_uid = self.thread.uid
            if not self.title:
                self.title = self.thread.title

            if not self.number:
                self.number = self.thread.number

            if not self.company_title:
                self.company_title = self.thread.company and self.thread.company.title

            if self.thread.t_model:
                self.t_code = self.thread.t_model.t_type.code

        if self.route:
            if not self.route_uid:
                self.route_uid = self.route.route_uid

    class Meta:
        index_together = [['zoom', 'lng', 'lat']]
        app_label = 'mapping'


class LiveMapObject(models.Model):
    """Движущийся объект на карте"""

    thread = models.ForeignKey('www.RThread', null=False)
    data = models.TextField(null=False)

    lng = models.FloatField(null=False, db_index=True)
    lat = models.FloatField(null=False, db_index=True)

    departure = models.DateTimeField(null=False, db_index=True)
    arrival = models.DateTimeField(null=False, db_index=True)

    class Meta:
        abstract = True

    def current_position(self, now):
        times, coords = self.get_data()

        return interpolate_coords(now, times, coords)

    def set_data(self, times, coords):
        self._times = times
        self._coords = coords

        self.data = '[%s]' % ",".join("[%.3f,%.6f,%.6f]" % (t, p[0], p[1])
                                      for t, p in izip(times, coords))

    def get_data(self):
        if not hasattr(self, '_data'):
            data = np.array(json.loads(self.data))

            times = data[..., 0]
            coords = data[..., 1:]

            self._data = times, coords

        return self._data

    def get_json_data(self, now, future_now):
        """Отрезаем кусок для отображения"""

        times, coords = self.get_data()

        # copypasted from fill_maptrain
        left_index = np.searchsorted(times, now, side='right')
        right_index = np.searchsorted(times, future_now, side='left')

        chunk_coords = coords[max(left_index - 1, 0):right_index + 1]
        chunk_times = times[max(left_index - 1, 0):right_index + 1]

        return list((t, p[0], p[1]) for t, p in izip(chunk_times, chunk_coords))


class Train(LiveMapObject):
    TIMESTAMP_ID = 'maptrain'
    EXPIRE_TIMESTAMP_ID = 'maptrainexpire'  # max 16 chars (common.models.timestamp.Timestamp)

    #
    # Мимикрируем под RThreadSegment для filter800suburban
    #
    @property
    def number(self):
        return self.thread.number

    @property
    def t_type(self):
        return self.thread.t_type

    class Meta:
        app_label = 'mapping'


class LiveBus(LiveMapObject):
    TIMESTAMP_ID = 'mapbus'
    EXPIRE_TIMESTAMP_ID = 'mapbusexpire'

    class Meta:
        app_label = 'mapping'


class RouteMapBlacklist(models.Model):
    number = TrimmedCharField(verbose_name=_(u'номер рейса'), max_length=100,
                              null=False, blank=False, unique=True)

    _numbers = None

    @classmethod
    def precache_numbers(cls):
        cls._numbers = set(entry.number for entry in cls.objects.all())

    @classmethod
    def clear_numbers_cache(cls):
        cls._numbers = None

    @classmethod
    def get_numbers_set(cls):
        if cls._numbers is None:
            cls.precache_numbers()
        return cls._numbers

    @classmethod
    def is_thread_mapped(cls, thread):
        if thread.t_type_id not in {TransportType.TRAIN_ID, TransportType.SUBURBAN_ID,
                                    TransportType.BUS_ID, TransportType.WATER_ID}:
            return False

        if thread.number in cls.get_numbers_set():
            return False

        # RASP-4111 - Скрывать поезда от ОАГ на карте
        if thread.supplier.code == 'oag':
            return False

        return True

    def __unicode__(self):
        return self.number

    class Meta:
        verbose_name = _(u'рейс, запрещенный к отображению на карте')
        verbose_name_plural = _(u'рейсы, запрещенные к отображению на карте')
        app_label = 'www'
