# -*- coding: utf-8 -*-

import base64
import logging
import math
import struct
from collections import namedtuple
from datetime import timedelta
from itertools import groupby, izip

import numpy as np
import pytz

from travel.rasp.library.python.common23.date import environment

from mapping.drawers import TrainPathDrawer, LimePathDrawer
from mapping.generators.utils import INT_PRECISION, FLOAT_PRECISION


log = logging.getLogger(__name__)


PathSegment = namedtuple('PathSegment', ['arc', 'departure', 'arrival'])
PathStation = namedtuple('PathStation', ['station', 'coords', 'arrival', 'departure', 'is_stop'])


class PathData(object):
    def __init__(self, path_arcs, path_stations, position, t_type):
        self.path_arcs = path_arcs
        self.path_stations = path_stations
        self.position = position

        self.first = None
        self.last = None

        self.t_type = t_type

    def __json__(self):
        rv = {
            'segments': [
                [decode_polyline(polyline), levels]
                for polyline, levels in self.path_arcs
            ],
            'stations': [
                [
                    path_station.coords,
                    path_station.station.L_title(),
                    path_station.arrival and path_station.arrival.strftime("%H:%M"),
                    path_station.departure and path_station.departure.strftime("%H:%M"),
                    path_station.station.id if not path_station.station.is_foreign() else None,  # RASP-4259
                ] for path_station in self.path_stations if path_station.is_stop
            ],
            'position': self.position,
            'transportType': self.t_type,
        }

        if self.first and self.last:
            rv.update({
                'first': self.first,
                'last': self.last,
            })

        return rv


def place_on_segment(segment, position):
    # координаты кривой
    iterator = (struct.unpack_from("<qq", segment, offset) for offset in xrange(0, len(segment), 16))

    lng0, lat0 = iterator.next()  # координаты начальной точки

    deltas = list(iterator)  # сдвиги относительно предыдущих точек

    # Длины сегментов
    segment_lengths = [math.sqrt(lngdelta ** 2 + latdelta ** 2) for lngdelta, latdelta in deltas]

    curve_length = sum(segment_lengths)

    current_length = 0  # длина сегмента в процессе поиска по нему
    current_position = lng0, lat0  # позиция в процессе поиска по сегменту

    curve_position = position * curve_length

    for delta, segment_length in izip(deltas, segment_lengths):
        if current_length <= curve_position <= (current_length + segment_length):
            # Отрезок превратился в точку
            if segment_length == 0:
                return current_position

            # пропорциональное положение на отрезке
            segment_position = (curve_position - current_length) / segment_length

            return [(c + d * segment_position) / FLOAT_PRECISION for c, d in zip(current_position, delta)]

        current_position = tuple(c + d for c, d in zip(current_position, delta))
        current_length += segment_length

    return [c / FLOAT_PRECISION for c in current_position]


def get_polyline_point(polyline, i):
    size = len(polyline)

    if i < 0:
        i = size - i

    if i < 0 or i >= size / 16:
        raise IndexError("polyline point index out of range")

    offset = i * 16

    int_coords = struct.unpack_from("<qq", polyline, offset)  # координаты кривой

    return [c / FLOAT_PRECISION for c in int_coords]


def get_current_position(naive_start_dt, path_arcs, path_stations):
    now = environment.now_aware()

    for i, path_arc in enumerate(path_arcs):
        departure = path_stations[i].departure
        arrival = path_stations[i + 1].arrival

        if departure < now < arrival:
            # поезд попал в промежуток между станциями, вычисляем позицию на кривой

            # относительное положение на длине кривой
            polyline_position = (
                (now - departure).total_seconds() /
                (arrival - departure).total_seconds()
            )

            return place_on_segment(path_arc[0], polyline_position)

    if path_stations[0].departure - timedelta(hours=1) <= now <= path_stations[0].departure:
        return path_stations[0].coords

    if path_stations[-1].arrival <= now <= path_stations[-1].arrival + timedelta(hours=0):
        return path_stations[-1].coords

    for path_station in path_stations[1:-1]:
        if path_station.arrival <= now <= path_station.departure:
            return path_station.coords

    return None


def draw_path(thread, thread_start_dt, path, first=None, last=None, zoom=None, clipper=None, time_zone=None):
    if thread_start_dt.tzinfo is None:
        naive_start_dt = thread_start_dt
    else:
        naive_start_dt = thread_start_dt.astimezone(thread.pytz).replace(tzinfo=None)

    if thread.route.t_type.code in ['train', 'suburban']:
        drawer = TrainPathDrawer(thread)
    else:
        drawer = LimePathDrawer()

    path_arcs = []
    path_stations = []

    for i, rts in enumerate(path):
        station_arrival = rts.get_loc_arrival_dt(naive_start_dt)
        station_departure = rts.get_loc_departure_dt(naive_start_dt)
        if time_zone:
            tz = pytz.timezone(time_zone)
            station_arrival = station_arrival.astimezone(tz) if station_arrival else None
            station_departure = station_departure.astimezone(tz) if station_departure else None

        if i > 0:
            rts1 = path[i - 1]
            rts2 = rts

            arc = drawer.get_arc(rts1.station_id, rts2.station_id)

            # При отсутствии сегмента не рисуем маршрут вообще
            if not arc:
                return None

            path_arcs.append(arc)

        station = PathStation(
            rts.station,
            drawer.get_station_coords(rts.station_id),
            station_arrival,
            station_departure,
            is_stop=station_arrival != station_departure and not rts.is_technical_stop
        )

        path_stations.append(station)

    position = get_current_position(naive_start_dt, path_arcs, path_stations)

    # Выборка лишних линий, возможно стоит перенести выше, чтобы быстрее и точнее
    # считать положение
    if zoom is not None:
        path_arcs = reduce_segments(path_arcs, zoom, clipper)

    path_data = PathData(
        path_arcs=path_arcs,
        path_stations=path_stations,
        position=position,
        t_type=thread.t_type.code
    )

    if first and last:
        path_data.first = drawer.get_station_coords(first.id)
        path_data.last = drawer.get_station_coords(last.id)

    return path_data


def walk_segments(segments):
    prev_coords = None

    for segment in segments:
        encoded, levels = segment

        curve = izip(
            (np.frombuffer(encoded, dtype=np.dtype('<i8')).reshape(-1, 2) / FLOAT_PRECISION).cumsum(axis=0),
            levels,
        )

        coords, level = curve.next()

        if prev_coords is None or np.any(prev_coords != coords):
            # Начало кривой не совпадает с концом предыдущей, начинаем новую
            yield None, None
            yield coords, 'A'

        for coords, level in curve:
            yield coords, level

        # Последняя точка этой кривой
        prev_coords = coords


def reduce_segments(segments, zoom, clipper=None):
    good_level = chr(ord('A') + zoom)

    reduced = []

    for coords, level in walk_segments(segments):
        if coords is None:
            # Начало новой кривой
            reduced_segment = []
            reduced_levels = []

            reduced.append((reduced_segment, reduced_levels))

            continue

        if level <= good_level:
            reduced_segment.append(coords)
            reduced_levels.append(level)

    for segment, levels in reduced:
        # Тут выкидываем лишнее и режем на куски
        if clipper:
            visibility = [0] * len(segment)

            for i in range(len(segment) - 1):
                x0, y0 = segment[i]
                x1, y1 = segment[i + 1]

                if clipper.test((x0, y0), (x1, y1)):
                    visibility[i] = visibility[i + 1] = 1
        else:
            visibility = [1] * len(segment)

        for v, points in groupby(izip(visibility, segment, levels), lambda t: t[0]):
            # Невидимые пропускаем
            if not v:
                continue

            prev_coords = [0, 0]
            deltas, levels = [], []

            for visible, coords, level in points:
                delta = coords[0] - prev_coords[0], coords[1] - prev_coords[1]

                deltas.append(delta)
                levels.append(level)

                prev_coords = coords

            # Фиксим конечные уровни
            levels[0] = levels[-1] = 'A'

            yield (
                (np.array(deltas) * INT_PRECISION).astype('<i8').tostring('C'),
                "".join(levels)
            )


def b64encode_segments(segments):
    return [
        [
            base64.urlsafe_b64encode(polyline),
            levels,
        ]
        for polyline, levels in segments
    ]


def decode_polyline(binary):
    """ДеКодирование ломанных из бинарного формата Я.Карт"""
    return (np.fromstring(binary, dtype=np.dtype('<i8')).reshape(-1, 2) / FLOAT_PRECISION).\
        cumsum(axis=0).tolist()
