# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from itertools import groupby

from django.db import models
from django.db.models import Q
from django.db import transaction
from django.utils.translation import ugettext_lazy as _

from common.models.schedule import Route, RThread
from common.models_utils import fetch_related


class ThreadSearchResult(object):
    """
    Значение True признака is_special_train означает, что переданная поисковая
    строка, - это начало названия найденных фирменных поездов. При этом нитки и
    маршруты будут содержать только данные по фирменным поездам.
    """

    def __init__(self, routes=None, threads=None, is_special_train=False):
        self.threads = threads or []
        self.routes = list(set(routes or []))
        self.is_special_train = is_special_train

        if self.routes and not self.threads:
            self.threads = get_threads_from_routes(self.routes)


class RouteNumberIndex(models.Model):
    u"""Слитные номера для поиска по рейсам"""
    number = models.CharField(verbose_name=_(u'просто Номер'), max_length=100, db_index=True,
                              null=False, blank=False, default=u"")
    reversed_number = models.CharField(verbose_name=_(u'перевернутый номер'), max_length=100,
                                       db_index=True, editable=False, null=False, blank=False)
    # title_special = models.CharField(verbose_name=_(u'фирменное название'), max_length=200,
    #                                  db_index=True, editable=False, null=False, blank=False, default='')
    route_uid = models.CharField(verbose_name=_(u"route_uid"),
                                 help_text=_(u"идентификатор для однозначного определения рейса из импорта"),
                                 max_length=100, null=False, blank=False, db_index=True)
    route = models.ForeignKey(Route, verbose_name=_(u"маршрут"), null=False,
                              blank=False)

    class Meta:
        app_label = 'www'
        verbose_name = _(u"индекс запись маршрут")
        verbose_name_plural = _(u"слитные номера для поиска по рейсам")

    @classmethod
    @transaction.atomic
    def rebuild_index(cls, partial=False):
        if partial:
            thread_filter = Q(changed=True, route__hidden=False)
            cls.objects.filter(route__rthread__changed=True).delete()
            cls.objects.filter(route__hidden=True).delete()
        else:
            cls.objects.all().delete()
            thread_filter = Q(route__hidden=False)

        def zero_trim(number):
            return number.lstrip(u'0')

        for route_id, thread_iter in groupby(RThread.objects.filter(thread_filter).select_related('route'),
                                             lambda t: t.route_id):
            thread = next(thread_iter)
            if thread.number:
                thread.reversed_number = thread.number[::-1]
                thread.save()

                RouteNumberIndex.objects.create(number=zero_trim(thread.number),
                                                reversed_number=thread.number[::-1],
                                                route_uid=thread.route.route_uid,
                                                route_id=route_id)

        for route_id, thread_iter in groupby(
            RThread.objects.filter(thread_filter, t_type__code='plane').select_related('company', 'route'),
            lambda t: t.route_id
        ):
            thread = next(thread_iter)
            if thread.number:
                numbers = set()
                if thread.company_id:
                    try:
                        flight_number = thread.number.split()[1].lstrip(u'0')
                    except IndexError:
                        pass
                    else:
                        for attr_name in ('sirena_id', 'iata', 'icao'):
                            code = getattr(thread.company, attr_name)
                            if code:
                                number = code + flight_number
                                numbers.add(number)

                numbers.add(thread.number)

                for number in numbers:
                    RouteNumberIndex.objects.create(number=zero_trim(number),
                                                    reversed_number=number[::-1],
                                                    route_uid=thread.route.route_uid,
                                                    route_id=route_id)


def get_threads_from_routes(routes):
    routes_by_id = dict((route.id, route) for route in routes)

    threads = list(RThread.objects.filter(route__in=routes, type=1).order_by())

    fetch_related(threads, 'schedule_plan', 'company', model=RThread)

    for thread in threads:
        thread.route = routes_by_id[thread.route_id]

    return threads
